(function() {
  const { registerBlockType } = wp.blocks;
  const { TextareaControl, SelectControl, PanelBody } = wp.components;
  const { InspectorControls } = wp.blockEditor || wp.editor;

  const LANGS = [
    { label: 'Bash / Shell', value: 'bash' },
    { label: 'PHP', value: 'php' },
    { label: 'HTML', value: 'html' },
    { label: 'JavaScript (JS)', value: 'js' },
    { label: 'CSS', value: 'css' },
    { label: 'ini', value: 'ini' },
    { label: 'YAML', value: 'yaml' },
    { label: 'Dockerfile', value: 'dockerfile' },
    // DÜZELTME: Bu satırın sonuna eksik olan virgül eklendi.
    { label: 'JSON', value: 'json' }
  ];

  // Otomatik girinti (auto-indent) fonksiyonu
  function handleIndent(e) {
    if (e.key === 'Enter') {
      e.preventDefault();
      const { target } = e;
      const { value, selectionStart, selectionEnd } = target;

      // Geçerli satırın başını bul
      let lineStart = selectionStart;
      while (lineStart > 0 && value[lineStart - 1] !== '\n') {
        lineStart--;
      }
      
      // Geçerli satırın girintisini (boşluk/tab) bul
      let indent = '';
      let i = lineStart;
      while (value[i] === ' ' || value[i] === '\t') {
        indent += value[i];
        i++;
      }
      
      // Yeni değeri oluştur: (önceki kısım) + (yeni satır) + (girinti) + (sonraki kısım)
      const newValue = value.substring(0, selectionStart) + '\n' + indent + value.substring(selectionEnd);
      
      // Bloğun 'code' attribute'unu güncelle
      const { setAttributes } = this;
      setAttributes({ code: newValue });
      
      // İmleç pozisyonunu yeni satırda doğru yere ayarla
      const newCursorPos = selectionStart + 1 + indent.length;
      requestAnimationFrame(() => {
        target.selectionStart = newCursorPos;
        target.selectionEnd = newCursorPos;
        // Kodu manuel olarak güncellediğimiz için Prism'i tekrar tetikle
        if (window.Prism) requestAnimationFrame(() => window.Prism.highlightAll());
      });
    }
  }

  registerBlockType('kupeli/code-plus', {
    edit: (props) => {
      const { attributes: { language, code }, setAttributes } = props;
      
      // handleIndent fonksiyonunu props ile bağla
      const boundHandleIndent = handleIndent.bind({ setAttributes });

      return (
        wp.element.createElement("div", { className: "kupeli-code-block is-editor" },
          wp.element.createElement(InspectorControls, null,
            wp.element.createElement(PanelBody, { title: "Code+ Ayarları" },
              wp.element.createElement(SelectControl, {
                label: "Kod Dili",
                value: language,
                options: LANGS,
                onChange: (val) => setAttributes({ language: val })
              })
            )
          ),
          wp.element.createElement("div", { className: "code-header" },
            wp.element.createElement("span", { className: "lang" }, (language || 'text').toUpperCase()),
            wp.element.createElement("button", { className: "copy-btn", type: "button", onClick: (e) => {
              const txt = code || '';
              if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(txt).then(()=>{
                  e.target.textContent = '✅ Kopyalandı'; setTimeout(()=> e.target.textContent='📋 Kopyala',1500);
                });
              } else {
                const ta = document.createElement('textarea'); ta.value = txt; document.body.appendChild(ta);
                ta.select(); document.execCommand('copy'); document.body.removeChild(ta);
                e.target.textContent = '✅ Kopyalandı'; setTimeout(()=> e.g.textContent='📋 Kopyala',1500);
              }
            }}, "📋 Kopyala")
          ),
          wp.element.createElement("div", { className: "editor-flex" },
            wp.element.createElement(TextareaControl, {
              label: "Kod",
              value: code,
              onChange: (val) => {
                setAttributes({ code: val });
                if (window.Prism) requestAnimationFrame(()=>window.Prism.highlightAll());
              },
              onKeyDown: boundHandleIndent
            }),
            wp.element.createElement("pre", { className: "line-numbers" },
              wp.element.createElement("code", { className: "language-" + (language || 'text') }, code)
            )
          )
        )
      );
    },
    save: (props) => {
      const { language, code } = props.attributes;
      const safe = (code || '').replace(/&/g, '&amp;').replace(/</g, '&lt;');
      return wp.element.createElement("div", { className: "kupeli-code-block" },
        wp.element.createElement("div", { className: "code-header" },
          wp.element.createElement("span", { className: "lang" }, (language || 'text').toUpperCase() ),
          wp.element.createElement("button", { className: "copy-btn", type: "button" }, "📋 Kopyala")
        ),
        wp.element.createElement("pre", { className: "line-numbers" },
          wp.element.createElement("code", { className: "language-" + (language || 'text') }, safe)
        )
      );
    }
  });
})();