<?php
/**
 * WP Security Fortress - Live Traffic
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WP_Security_Fortress_Live_Traffic {

    private $option_key        = 'wpf_live_traffic_logs';
    private $max_entries       = 100;
    private $geo_cache_hours   = 24;
    private $geo_cache_prefix  = 'wpf_geo_';
    private $auto_refresh_secs = 15;

    public function __construct() {
        add_action( 'init', array( $this, 'capture_request' ), 1 );
        add_action( 'admin_menu', array( $this, 'register_admin_page' ) );
        add_action( 'admin_init', array( $this, 'handle_clear_logs' ) );
    }

    public function capture_request() {
        if ( ( defined( 'DOING_CRON' ) && DOING_CRON ) || ( defined( 'WP_CLI' ) && WP_CLI ) ) return;

        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
        if ( $request_uri === '' ) return;

        $method = isset( $_SERVER['REQUEST_METHOD'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : 'GET';
        $ip     = $this->get_ip();
        $ua     = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';

        $username = '';
        if ( is_user_logged_in() ) {
            $u = wp_get_current_user();
            if ( $u instanceof WP_User ) $username = $u->user_login;
        }

        $geo = $this->get_geo_for_ip( $ip );

        $entry = array(
            'time'         => time(),
            'ip'           => $ip,
            'method'       => $method,
            'uri'          => $request_uri,
            'ua'           => $ua,
            'username'     => $username,
            'country_code' => isset( $geo['country_code'] ) ? $geo['country_code'] : '',
            'country'      => isset( $geo['country'] ) ? $geo['country'] : '',
            'city'         => isset( $geo['city'] ) ? $geo['city'] : '',
        );

        $logs = get_option( $this->option_key, array() );
        if ( ! is_array( $logs ) ) $logs = array();

        array_unshift( $logs, $entry );

        if ( count( $logs ) > $this->max_entries ) {
            $logs = array_slice( $logs, 0, $this->max_entries );
        }

        update_option( $this->option_key, $logs, 'no' );
    }

    private function get_ip() {
        if ( ! empty( $_SERVER['HTTP_CF_CONNECTING_IP'] ) ) {
            $ip = wp_unslash( $_SERVER['HTTP_CF_CONNECTING_IP'] );
        } elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
            $parts = explode( ',', wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
            $ip    = trim( $parts[0] );
        } elseif ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
            $ip = wp_unslash( $_SERVER['REMOTE_ADDR'] );
        } else {
            $ip = '0.0.0.0';
        }
        $ip = sanitize_text_field( $ip );
        return filter_var( $ip, FILTER_VALIDATE_IP ) ? $ip : '0.0.0.0';
    }

    private function get_geo_for_ip( $ip ) {
        $ip = trim( $ip );
        if ( $ip === '' || $ip === '0.0.0.0' ) return array();

        $cache_key = $this->geo_cache_prefix . md5( $ip );
        $cached    = get_transient( $cache_key );
        if ( is_array( $cached ) ) return $cached;

        $response = wp_remote_get( 'https://ipwho.is/' . rawurlencode( $ip ), array( 'timeout' => 3 ) );
        if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) return array();

        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! is_array( $data ) || empty( $data['success'] ) ) return array();

        $geo = array(
            'country_code' => $data['country_code'] ?? '',
            'country'      => $data['country'] ?? '',
            'city'         => $data['city'] ?? '',
        );

        set_transient( $cache_key, $geo, HOUR_IN_SECONDS * $this->geo_cache_hours );
        return $geo;
    }

    public function register_admin_page() {
        add_submenu_page(
            'wp-security-fortress',
            'Security Fortress - Live Traffic',
            'Live Traffic',
            'manage_options',
            'wp-security-fortress-live-traffic',
            array( $this, 'render_admin_page' )
        );
    }

    public function handle_clear_logs() {
        if ( ! isset( $_GET['wpf_lt_action'], $_GET['_wpnonce'] ) || 'clear' !== $_GET['wpf_lt_action'] ) return;
        if ( ! current_user_can( 'manage_options' ) ) return;
        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'wpf_lt_clear_logs' ) ) return;

        update_option( $this->option_key, array(), 'no' );
        wp_safe_redirect( admin_url( 'admin.php?page=wp-security-fortress-live-traffic&msg=cleared' ) );
        exit;
    }

    public function render_admin_page() {
        if ( ! current_user_can( 'manage_options' ) ) return;

        $logs = get_option( $this->option_key, array() );
        if ( ! is_array( $logs ) ) $logs = array();

        $clear_url = wp_nonce_url( admin_url( 'admin.php?page=wp-security-fortress-live-traffic&wpf_lt_action=clear' ), 'wpf_lt_clear_logs' );
        $auto_refresh_enabled = empty( $_GET['no_refresh'] );
        ?>
        <div class="wrap">
            <h1>WP Security Fortress - Live Traffic</h1>
            <?php if ( isset( $_GET['msg'] ) && 'cleared' === $_GET['msg'] ) : ?>
                <div class="notice notice-success is-dismissible"><p>Loglar temizlendi.</p></div>
            <?php endif; ?>

            <p>
                <?php if ( $auto_refresh_enabled ) : ?>
                    <span style="color:green;">Otomatik yenileme aktif (15sn).</span> | <a href="<?php echo esc_url( add_query_arg( 'no_refresh', '1' ) ); ?>">Durdur</a>
                <?php else : ?>
                    <span style="color:gray;">Otomatik yenileme kapalı.</span> | <a href="<?php echo esc_url( remove_query_arg( 'no_refresh' ) ); ?>">Aç</a>
                <?php endif; ?>
                <a href="<?php echo esc_url( $clear_url ); ?>" class="button button-secondary" style="float:right;" onclick="return confirm('Emin misiniz?');">Logları Temizle</a>
            </p>

            <table class="widefat fixed striped">
                <thead><tr><th>Tarih</th><th>IP</th><th>Konum</th><th>Kullanıcı</th><th>Metod</th><th>URL</th></tr></thead>
                <tbody>
                    <?php if ( empty( $logs ) ) : ?>
                        <tr><td colspan="6">Kayıt yok.</td></tr>
                    <?php else : foreach ( $logs as $row ) : 
                        $ip = esc_attr($row['ip']??'');
                    ?>
                        <tr>
                            <td><?php echo date_i18n( 'd.m.Y H:i', $row['time'] ?? time() ); ?></td>
                            <td>
                                <a href="#" class="wpf-ip-trigger" data-ip="<?php echo $ip; ?>" style="font-weight:600;text-decoration:none;">
                                    <?php echo esc_html($ip); ?>
                                    <span class="dashicons dashicons-search" style="font-size:14px;vertical-align:middle;color:#aaa;"></span>
                                </a>
                            </td>
                            <td>
                                <?php if(!empty($row['country_code'])): ?>
                                    <img src="https://cdn.ipwhois.io/flags/<?php echo strtolower($row['country_code']); ?>.svg" style="height:12px;"> 
                                <?php endif; ?>
                                <?php echo esc_html( ($row['city']??'') . ' ' . ($row['country']??'') ); ?>
                            </td>
                            <td><?php echo esc_html($row['username'] ?: 'guest'); ?></td>
                            <td><?php echo esc_html($row['method']??''); ?></td>
                            <td><code style="font-size:10px;"><?php echo esc_html($row['uri']??''); ?></code></td>
                        </tr>
                    <?php endforeach; endif; ?>
                </tbody>
            </table>
        </div>

        <div id="wpf-ip-modal" class="wpf-modal">
            <div class="wpf-modal-content">
                <span class="wpf-close">&times;</span>
                <div id="wpf-modal-header" style="display:flex;align-items:center;border-bottom:1px solid #eee;padding-bottom:10px;margin-bottom:15px;">
                    <h2 style="margin:0;font-size:18px;">IP Detayları</h2>
                </div>
                <div id="wpf-modal-body">
                    <div class="wpf-loader">Yükleniyor...</div>
                </div>
            </div>
        </div>

        <style>
            .wpf-modal { display: none; position: fixed; z-index: 99999; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.5); backdrop-filter: blur(2px); animation: wpfFadeIn 0.2s; }
            .wpf-modal-content { background-color: #fefefe; margin: 10% auto; padding: 25px; border: 1px solid #888; width: 90%; max-width: 450px; border-radius: 12px; box-shadow: 0 10px 25px rgba(0,0,0,0.2); position:relative; }
            .wpf-close { color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer; position:absolute; right:15px; top:10px; }
            .wpf-close:hover, .wpf-close:focus { color: black; text-decoration: none; cursor: pointer; }
            .wpf-detail-row { display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0; }
            .wpf-detail-row:last-child { border-bottom: none; }
            .wpf-label { font-weight: 600; color: #555; }
            .wpf-value { color: #222; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif; text-align:right;}
            .wpf-loader { text-align: center; color: #666; padding: 20px; }
            @keyframes wpfFadeIn { from {opacity: 0;} to {opacity: 1;} }
        </style>

        <script>
        document.addEventListener('DOMContentLoaded', function() {
            var modal = document.getElementById("wpf-ip-modal");
            var span = document.getElementsByClassName("wpf-close")[0];
            var body = document.getElementById("wpf-modal-body");
            var header = document.getElementById("wpf-modal-header");

            document.querySelectorAll('.wpf-ip-trigger').forEach(function(el) {
                el.addEventListener('click', function(e) {
                    e.preventDefault();
                    var ip = this.getAttribute('data-ip');
                    
                    modal.style.display = "block";
                    body.innerHTML = '<div class="wpf-loader">Veriler çekiliyor...<br>'+ip+'</div>';
                    
                    // IP verisini çek
                    fetch('https://ipwho.is/' + ip)
                    .then(response => response.json())
                    .then(data => {
                        if(!data.success) {
                            body.innerHTML = '<p style="color:red;">Hata: ' + (data.message || 'Veri alınamadı') + '</p>';
                            return;
                        }

                        var html = '';
                        // Başlığa bayrak ekle
                        var flag = data.flag && data.flag.img ? '<img src="'+data.flag.img+'" style="height:24px;margin-right:10px;border-radius:3px;">' : '';
                        header.innerHTML = '<div style="display:flex;align-items:center;">' + flag + '<h2 style="margin:0;">' + ip + '</h2></div>';

                        // Detay satırları
                        html += '<div class="wpf-detail-row"><span class="wpf-label">Ülke:</span><span class="wpf-value">' + data.country + ' (' + data.country_code + ')</span></div>';
                        html += '<div class="wpf-detail-row"><span class="wpf-label">Şehir:</span><span class="wpf-value">' + data.city + '</span></div>';
                        html += '<div class="wpf-detail-row"><span class="wpf-label">Bölge:</span><span class="wpf-value">' + data.region + '</span></div>';
                        
                        if(data.connection && data.connection.isp) {
                            html += '<div class="wpf-detail-row"><span class="wpf-label">ISP:</span><span class="wpf-value">' + data.connection.isp + '</span></div>';
                        }
                        if(data.connection && data.connection.org) {
                            html += '<div class="wpf-detail-row"><span class="wpf-label">Org:</span><span class="wpf-value">' + data.connection.org + '</span></div>';
                        }
                        if(data.timezone && data.timezone.id) {
                            html += '<div class="wpf-detail-row"><span class="wpf-label">Zaman Dilimi:</span><span class="wpf-value">' + data.timezone.current_time + '</span></div>';
                        }

                        // Google Maps Linki
                        var lat = data.latitude;
                        var lon = data.longitude;
                        html += '<div style="margin-top:15px;text-align:center;"><a href="https://www.google.com/maps/search/?api=1&query=' + lat + ',' + lon + '" target="_blank" class="button button-primary">Haritada Göster</a></div>';

                        body.innerHTML = html;
                    })
                    .catch(err => {
                        body.innerHTML = '<p style="color:red;">Bağlantı hatası.</p>';
                    });
                });
            });

            span.onclick = function() { modal.style.display = "none"; }
            window.onclick = function(event) {
                if (event.target == modal) { modal.style.display = "none"; }
            }
        });
        
        <?php if ( $auto_refresh_enabled ) : ?>
            setTimeout(function(){ if(document.visibilityState==='visible') window.location.reload(); }, 15000);
        <?php endif; ?>
        </script>
        <?php
    }
}
new WP_Security_Fortress_Live_Traffic();