<?php
/**
 * WP Security Fortress - Session Management
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WP_Security_Fortress_Sessions {

    private $option_key      = 'wpf_session_manager';
    private $cookie_name     = 'wpf_session_id';
    private $default_options = array(
        'enabled'           => 1,
        'max_sessions'      => 1,
        'idle_timeout_min'  => 30,
        'absolute_hours'    => 24,
        'limit_behavior'    => 'kill_old',
    );

    public function __construct() {
        add_action( 'wp_login', array( $this, 'handle_login' ), 10, 2 );
        add_action( 'init', array( $this, 'enforce_session' ), 1 );
        add_action( 'admin_menu', array( $this, 'register_admin_page' ) );
        add_action( 'admin_init', array( $this, 'handle_admin_actions' ) );
    }

    private function get_options() {
        $opts = get_option( $this->option_key, array() );
        return wp_parse_args( $opts, $this->default_options );
    }

    private function save_options( $opts ) {
        $opts = wp_parse_args( $opts, $this->default_options );
        update_option( $this->option_key, $opts );
    }

    private function get_user_sessions( $user_id ) {
        $sessions = get_user_meta( $user_id, '_wpf_sessions', true );
        return is_array( $sessions ) ? $sessions : array();
    }

    private function save_user_sessions( $user_id, $sessions ) {
        if ( empty( $sessions ) ) {
            delete_user_meta( $user_id, '_wpf_sessions' );
        } else {
            update_user_meta( $user_id, '_wpf_sessions', $sessions );
        }
    }

    private function create_session( $user_id ) {
        $session_id = wp_generate_password( 32, false, false );
        $now        = time();
        $ip  = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $ua  = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';

        $sessions = $this->get_user_sessions( $user_id );
        $sessions[ $session_id ] = array(
            'created'       => $now,
            'last_activity' => $now,
            'ip'            => $ip,
            'ua'            => $ua,
        );
        $this->save_user_sessions( $user_id, $sessions );
        $this->set_session_cookie( $session_id );
        return $session_id;
    }

    private function set_session_cookie( $session_id ) {
        if ( headers_sent() ) return;
        $secure = is_ssl();
        
        if ( PHP_VERSION_ID < 70300 ) {
            setcookie( $this->cookie_name, $session_id, time() + DAY_IN_SECONDS * 2, COOKIEPATH . '; samesite=Lax', COOKIE_DOMAIN, $secure, true );
        } else {
            setcookie( $this->cookie_name, $session_id, array(
                'expires'  => time() + DAY_IN_SECONDS * 2,
                'path'     => COOKIEPATH,
                'domain'   => COOKIE_DOMAIN,
                'secure'   => $secure,
                'httponly' => true,
                'samesite' => 'Lax',
            ));
        }
    }

    private function clear_session_cookie() {
        if ( headers_sent() ) return;
        setcookie( $this->cookie_name, '', time() - HOUR_IN_SECONDS, COOKIEPATH, COOKIE_DOMAIN );
    }

    private function get_current_session_id() {
        if ( empty( $_COOKIE[ $this->cookie_name ] ) ) return '';
        $sid = sanitize_text_field( wp_unslash( $_COOKIE[ $this->cookie_name ] ) );
        return preg_match( '/^[A-Za-z0-9]+$/', $sid ) ? $sid : '';
    }

    private function is_session_expired( $session, $opts ) {
        $now = time();
        $idle_timeout = max( 1, intval( $opts['idle_timeout_min'] ) ) * MINUTE_IN_SECONDS;
        $absolute     = max( 1, intval( $opts['absolute_hours'] ) ) * HOUR_IN_SECONDS;

        $created       = isset( $session['created'] ) ? intval( $session['created'] ) : $now;
        $last_activity = isset( $session['last_activity'] ) ? intval( $session['last_activity'] ) : $created;

        if ( ( $now - $last_activity ) > $idle_timeout ) return true;
        if ( ( $now - $created ) > $absolute ) return true;
        return false;
    }

    public function handle_login( $user_login, $user ) {
        $opts = $this->get_options();
        if ( empty( $opts['enabled'] ) ) return;

        $user_id  = $user->ID;
        $sessions = $this->get_user_sessions( $user_id );

        foreach ( $sessions as $sid => $session ) {
            if ( $this->is_session_expired( $session, $opts ) ) {
                unset( $sessions[ $sid ] );
            }
        }

        $max = max( 1, intval( $opts['max_sessions'] ) );

        if ( count( $sessions ) >= $max ) {
            if ( 'block_new' === $opts['limit_behavior'] ) {
                return;
            } else {
                uasort( $sessions, function ( $a, $b ) {
                    $ca = isset( $a['created'] ) ? $a['created'] : 0;
                    $cb = isset( $b['created'] ) ? $b['created'] : 0;
                    return ( $ca === $cb ) ? 0 : ( ($ca < $cb) ? -1 : 1 );
                });
                while ( count( $sessions ) >= $max ) {
                    array_shift( $sessions );
                }
            }
        }

        $this->save_user_sessions( $user_id, $sessions );
        $this->create_session( $user_id );
    }

    public function enforce_session() {
        $opts = $this->get_options();
        if ( empty( $opts['enabled'] ) ) return;

        if ( is_user_logged_in() ) {
            $user_id    = get_current_user_id();
            $sessions   = $this->get_user_sessions( $user_id );
            $session_id = $this->get_current_session_id();

            if ( $session_id && isset( $sessions[ $session_id ] ) ) {
                if ( $this->is_session_expired( $sessions[ $session_id ], $opts ) ) {
                    unset( $sessions[ $session_id ] );
                    $this->save_user_sessions( $user_id, $sessions );
                    $this->force_logout_with_message( __( 'Oturum süreniz dolduğu için çıkış yapıldı.', 'wp-security-fortress' ) );
                } else {
                    $sessions[ $session_id ]['last_activity'] = time();
                    $this->save_user_sessions( $user_id, $sessions );
                }
            } else {
                if ( ! empty( $sessions ) ) {
                    $this->force_logout_with_message( __( 'Oturumunuz başka bir cihazdan sonlandırıldı.', 'wp-security-fortress' ) );
                } else {
                    $this->create_session( $user_id );
                }
            }
        } else {
            if ( $this->get_current_session_id() ) {
                $this->clear_session_cookie();
            }
        }
    }

    private function force_logout_with_message( $message ) {
        $this->clear_session_cookie();
        wp_logout();
        wp_safe_redirect( add_query_arg( array( 'wpf_session_msg' => rawurlencode( wp_strip_all_tags( $message ) ) ), wp_login_url() ) );
        exit;
    }

    public function register_admin_page() {
        // GÜNCELLEME: add_options_page yerine add_submenu_page kullanıyoruz
        // Parent slug: wp-security-fortress (Ana menü slug'ı)
        add_submenu_page(
            'wp-security-fortress',
            'Security Fortress - Sessions',
            'Sessions',
            'manage_options',
            'wp-security-fortress-sessions',
            array( $this, 'render_admin_page' )
        );
    }

    public function handle_admin_actions() {
        if ( ! current_user_can( 'manage_options' ) ) return;

        if ( isset( $_POST['wpf_sessions_settings_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wpf_sessions_settings_nonce'] ) ), 'wpf_sessions_save' ) ) {
            $opts = $this->get_options();
            $opts['enabled']          = isset( $_POST['wpf_enabled'] ) ? 1 : 0;
            $opts['max_sessions']     = isset( $_POST['wpf_max_sessions'] ) ? max( 1, intval( $_POST['wpf_max_sessions'] ) ) : $opts['max_sessions'];
            $opts['idle_timeout_min'] = isset( $_POST['wpf_idle_timeout'] ) ? max( 1, intval( $_POST['wpf_idle_timeout'] ) ) : $opts['idle_timeout_min'];
            $opts['absolute_hours']   = isset( $_POST['wpf_absolute_hours'] ) ? max( 1, intval( $_POST['wpf_absolute_hours'] ) ) : $opts['absolute_hours'];
            $opts['limit_behavior']   = ( isset( $_POST['wpf_limit_behavior'] ) && in_array( $_POST['wpf_limit_behavior'], array( 'kill_old', 'block_new' ), true ) ) ? $_POST['wpf_limit_behavior'] : $opts['limit_behavior'];

            $this->save_options( $opts );

            // URL GÜNCELLEMESİ: admin.php
            wp_safe_redirect( add_query_arg( 'wpf_sess_msg', 'saved', admin_url( 'admin.php?page=wp-security-fortress-sessions' ) ) );
            exit;
        }

        if ( isset( $_GET['wpf_sess_action'], $_GET['_wpnonce'] ) && 'kill' === $_GET['wpf_sess_action'] ) {
            if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'wpf_sess_kill' ) ) return;

            $user_id    = isset( $_GET['user_id'] ) ? intval( $_GET['user_id'] ) : 0;
            $session_id = isset( $_GET['session_id'] ) ? sanitize_text_field( wp_unslash( $_GET['session_id'] ) ) : '';

            if ( $user_id > 0 && $session_id ) {
                $sessions = $this->get_user_sessions( $user_id );
                if ( isset( $sessions[ $session_id ] ) ) {
                    unset( $sessions[ $session_id ] );
                    $this->save_user_sessions( $user_id, $sessions );
                }
            }

            // URL GÜNCELLEMESİ: admin.php
            wp_safe_redirect( add_query_arg( 'wpf_sess_msg', 'killed', admin_url( 'admin.php?page=wp-security-fortress-sessions' ) ) );
            exit;
        }
    }

    public function render_admin_page() {
        if ( ! current_user_can( 'manage_options' ) ) return;

        $opts = $this->get_options();
        $users = get_users( array( 'fields' => array( 'ID', 'user_login', 'display_name' ) ) );
        $active_sessions = array();

        foreach ( $users as $user ) {
            $sessions = $this->get_user_sessions( $user->ID );
            if ( empty( $sessions ) ) continue;
            foreach ( $sessions as $sid => $session ) {
                $active_sessions[] = array(
                    'user_id' => $user->ID,
                    'login' => $user->user_login,
                    'name' => $user->display_name,
                    'sid' => $sid,
                    'session' => $session,
                );
            }
        }
        ?>
        <div class="wrap">
            <h1>WP Security Fortress - Session Management</h1>
            <?php if ( isset( $_GET['wpf_sess_msg'] ) ) : ?>
                <div class="notice notice-success is-dismissible"><p><?php echo ($_GET['wpf_sess_msg'] === 'saved') ? 'Ayarlar kaydedildi.' : 'Oturum sonlandırıldı.'; ?></p></div>
            <?php endif; ?>

            <h2>Ayarlar</h2>
            <form method="post" action="">
                <?php wp_nonce_field( 'wpf_sessions_save', 'wpf_sessions_settings_nonce' ); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Oturum Yönetimi</th>
                        <td><label><input type="checkbox" name="wpf_enabled" value="1" <?php checked( $opts['enabled'], 1 ); ?> /> Aktif Et</label></td>
                    </tr>
                    <tr>
                        <th scope="row">Maksimum Oturum</th>
                        <td><input type="number" name="wpf_max_sessions" min="1" value="<?php echo esc_attr( $opts['max_sessions'] ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Zaman Aşımı (Dk)</th>
                        <td><input type="number" name="wpf_idle_timeout" min="1" value="<?php echo esc_attr( $opts['idle_timeout_min'] ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Mutlak Süre (Saat)</th>
                        <td><input type="number" name="wpf_absolute_hours" min="1" value="<?php echo esc_attr( $opts['absolute_hours'] ); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Limit Davranışı</th>
                        <td>
                            <label><input type="radio" name="wpf_limit_behavior" value="kill_old" <?php checked( $opts['limit_behavior'], 'kill_old' ); ?> /> Eskiyi kapat</label><br>
                            <label><input type="radio" name="wpf_limit_behavior" value="block_new" <?php checked( $opts['limit_behavior'], 'block_new' ); ?> /> Yeniyi engelle</label>
                        </td>
                    </tr>
                </table>
                <?php submit_button( 'Ayarları Kaydet' ); ?>
            </form>

            <hr />
            <h2>Aktif Oturumlar</h2>
            <table class="widefat fixed striped">
                <thead><tr><th>Kullanıcı</th><th>IP / UA</th><th>Oluşturulma</th><th>Son İşlem</th><th>Durum</th><th>İşlem</th></tr></thead>
                <tbody>
                    <?php if ( empty( $active_sessions ) ) : ?>
                        <tr><td colspan="6">Aktif oturum yok.</td></tr>
                    <?php else : 
                        foreach ( $active_sessions as $row ) :
                            $kill_url = wp_nonce_url( add_query_arg( array( 'wpf_sess_action' => 'kill', 'user_id' => $row['user_id'], 'session_id' => $row['sid'] ), admin_url( 'admin.php?page=wp-security-fortress-sessions' ) ), 'wpf_sess_kill' );
                    ?>
                        <tr>
                            <td><strong><?php echo esc_html( $row['login'] ); ?></strong></td>
                            <td><?php echo esc_html( $row['session']['ip'] ?? '' ); ?><br><small><?php echo esc_html( substr($row['session']['ua'] ?? '', 0, 50) ); ?></small></td>
                            <td><?php echo date_i18n( 'd.m.Y H:i', $row['session']['created'] ); ?></td>
                            <td><?php echo date_i18n( 'd.m.Y H:i', $row['session']['last_activity'] ); ?></td>
                            <td><?php echo $this->is_session_expired($row['session'], $opts) ? '<span style="color:red">Süresi Dolmuş</span>' : '<span style="color:green">Aktif</span>'; ?></td>
                            <td><a href="<?php echo esc_url( $kill_url ); ?>" class="button button-small" onclick="return confirm('Emin misiniz?');">Sonlandır</a></td>
                        </tr>
                    <?php endforeach; endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }
}
new WP_Security_Fortress_Sessions();