<?php
/**
 * WP Security Fortress - SMTP Modülü
 * Gmail altyapısını kullanarak WordPress maillerini güvenli bir şekilde gönderir.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WP_Security_Fortress_SMTP {

    private $option_key = 'wpf_smtp_options';

    public function __construct() {
        // Menüye ekle
        add_action( 'admin_menu', array( $this, 'register_admin_page' ) );
        
        // PHPMailer ayarlarını değiştir (Asıl sihir burada)
        add_action( 'phpmailer_init', array( $this, 'configure_smtp' ) );
        
        // Test maili gönderme işlemi
        add_action( 'admin_init', array( $this, 'handle_test_email' ) );
    }

    /**
     * WordPress mail fonksiyonunu Gmail'e yönlendirir
     */
    public function configure_smtp( $phpmailer ) {
        $opts = get_option( $this->option_key, array() );

        // Eğer modül kapalıysa veya şifre yoksa karışma
        if ( empty( $opts['enabled'] ) || empty( $opts['password'] ) || empty( $opts['email'] ) ) {
            return;
        }

        $phpmailer->isSMTP();
        $phpmailer->Host       = 'smtp.gmail.com';
        $phpmailer->SMTPAuth   = true;
        $phpmailer->Port       = 587;
        $phpmailer->SMTPSecure = 'tls';
        
        $phpmailer->Username   = $opts['email'];
        $phpmailer->Password   = $opts['password'];
        
        // Gönderen Bilgileri
        $phpmailer->From       = $opts['email'];
        $phpmailer->FromName   = !empty( $opts['from_name'] ) ? $opts['from_name'] : get_bloginfo( 'name' );
    }

    /**
     * Admin Menüsü
     */
    public function register_admin_page() {
        add_submenu_page(
            'wp-security-fortress',
            'Security Fortress - SMTP',
            'SMTP Ayarları',
            'manage_options',
            'wp-security-fortress-smtp',
            array( $this, 'render_page' )
        );
    }

    /**
     * Test Maili İşleyicisi
     */
    public function handle_test_email() {
        if ( isset( $_POST['wpf_smtp_test'], $_POST['_wpnonce'] ) && check_admin_referer( 'wpf_smtp_test_action' ) ) {
            
            $to = sanitize_email( $_POST['test_email_recipient'] );
            if ( ! is_email( $to ) ) {
                return;
            }

            // Debug çıktısını yakalamak için
            global $phpmailer;
            if ( ! is_object( $phpmailer ) || ! is_a( $phpmailer, 'PHPMailer' ) ) {
                require_once ABSPATH . WPINC . '/class-phpmailer.php';
                require_once ABSPATH . WPINC . '/class-smtp.php';
                $phpmailer = new PHPMailer( true );
            }

            // Hata yakalama modunu aç
            $phpmailer->SMTPDebug = 2; // Client ve Server iletişimini göster
            ob_start(); // Çıktıyı tampona al
            
            $sent = wp_mail( $to, 'WP Security Fortress - Test Maili', "Merhaba,\n\nBu bir test mailidir. SMTP ayarlarınız başarıyla çalışıyor.\n\nZaman: " . current_time('mysql') );
            
            $debug_log = ob_get_clean(); // Tamponu değişkene at ve temizle

            if ( $sent ) {
                wp_safe_redirect( admin_url( 'admin.php?page=wp-security-fortress-smtp&msg=sent' ) );
            } else {
                // Hataları transient'e kaydet ki sayfada gösterebilelim
                set_transient( 'wpf_smtp_error_log', $debug_log, 60 );
                wp_safe_redirect( admin_url( 'admin.php?page=wp-security-fortress-smtp&msg=failed' ) );
            }
            exit;
        }

        // Ayarları Kaydetme İşlemi
        if ( isset( $_POST['wpf_smtp_save'], $_POST['_wpnonce'] ) && check_admin_referer( 'wpf_smtp_save_action' ) ) {
            $new_opts = array(
                'enabled'   => isset( $_POST['enabled'] ) ? 1 : 0,
                'email'     => sanitize_email( $_POST['email'] ),
                'password'  => sanitize_text_field( $_POST['password'] ), // Uygulama şifresi
                'from_name' => sanitize_text_field( $_POST['from_name'] ),
            );
            update_option( $this->option_key, $new_opts );
            wp_safe_redirect( admin_url( 'admin.php?page=wp-security-fortress-smtp&msg=saved' ) );
            exit;
        }
    }

    /**
     * Ayar Sayfası HTML
     */
    public function render_page() {
        $opts = get_option( $this->option_key, array( 'enabled' => 0, 'email' => '', 'password' => '', 'from_name' => '' ) );
        $error_log = get_transient( 'wpf_smtp_error_log' );
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-email-alt"></span> SMTP & Bildirim Ayarları (Gmail)</h1>
            
            <?php if ( isset( $_GET['msg'] ) ) : ?>
                <?php if ( $_GET['msg'] === 'saved' ) : ?>
                    <div class="notice notice-success is-dismissible"><p>Ayarlar kaydedildi.</p></div>
                <?php elseif ( $_GET['msg'] === 'sent' ) : ?>
                    <div class="notice notice-success is-dismissible"><p><strong>Başarılı!</strong> Test maili gönderildi. Lütfen gelen kutunuzu (ve spam klasörünü) kontrol edin.</p></div>
                <?php elseif ( $_GET['msg'] === 'failed' ) : ?>
                    <div class="notice notice-error"><p><strong>Hata:</strong> Mail gönderilemedi.</p></div>
                <?php endif; ?>
            <?php endif; ?>

            <?php if ( $error_log ) : ?>
                <div class="error" style="padding:15px; background:#ffebeb; border-left:4px solid #dc3232; margin-bottom:20px;">
                    <h3>Hata Kayıtları (Debug Log):</h3>
                    <pre style="white-space: pre-wrap; font-size:11px; color:#333;"><?php echo esc_html( $error_log ); ?></pre>
                    <p><em>Google "Bad Credentials" diyorsa: Uygulama şifresi yanlıştır. "Connection Timed Out" diyorsa sunucunun 587 portu kapalıdır.</em></p>
                </div>
                <?php delete_transient( 'wpf_smtp_error_log' ); ?>
            <?php endif; ?>

            <div style="display:flex; gap:20px; flex-wrap:wrap;">
                
                <div style="flex:1; min-width:300px; background:#fff; padding:20px; border:1px solid #ccd0d4; box-shadow:0 1px 4px rgba(0,0,0,0.05); border-radius:8px;">
                    <h2 style="margin-top:0;">Gmail Bağlantısı</h2>
                    <form method="post">
                        <?php wp_nonce_field( 'wpf_smtp_save_action' ); ?>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">Durum</th>
                                <td>
                                    <label class="switch">
                                        <input type="checkbox" name="enabled" value="1" <?php checked( $opts['enabled'], 1 ); ?>>
                                        Aktif Et
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Gmail Adresi</th>
                                <td>
                                    <input type="email" name="email" value="<?php echo esc_attr( $opts['email'] ); ?>" class="regular-text" placeholder="ornek@gmail.com" required>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Uygulama Şifresi</th>
                                <td>
                                    <input type="password" name="password" value="<?php echo esc_attr( $opts['password'] ); ?>" class="regular-text" placeholder="xxxx xxxx xxxx xxxx">
                                    <p class="description">Google Hesabım > Güvenlik > 2 Adımlı Doğrulama > Uygulama Şifreleri bölümünden aldığınız 16 haneli kod.</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Gönderici Adı</th>
                                <td>
                                    <input type="text" name="from_name" value="<?php echo esc_attr( $opts['from_name'] ); ?>" class="regular-text" placeholder="Örn: Docknology Güvenlik">
                                </td>
                            </tr>
                        </table>
                        
                        <p class="submit">
                            <button type="submit" name="wpf_smtp_save" class="button button-primary">Ayarları Kaydet</button>
                        </p>
                    </form>
                </div>

                <div style="flex:1; min-width:300px; background:#fff; padding:20px; border:1px solid #ccd0d4; box-shadow:0 1px 4px rgba(0,0,0,0.05); border-radius:8px; height:fit-content;">
                    <h2 style="margin-top:0;">Test Maili Gönder</h2>
                    <p>Ayarları kaydettikten sonra buradan test edebilirsiniz.</p>
                    
                    <form method="post">
                        <?php wp_nonce_field( 'wpf_smtp_test_action' ); ?>
                        <p>
                            <label>Alıcı E-posta:</label><br>
                            <input type="email" name="test_email_recipient" value="<?php echo esc_attr( wp_get_current_user()->user_email ); ?>" class="regular-text" required>
                        </p>
                        <button type="submit" name="wpf_smtp_test" class="button button-secondary">Test Maili Gönder</button>
                    </form>
                </div>

            </div>
        </div>
        <?php
    }
}

new WP_Security_Fortress_SMTP();