<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WP_Security_Fortress {

    private string $option_key      = 'wp_sll_options';
    private string $logs_option_key = 'wp_sll_logs';
    private int $max_logs           = 500;
    
    private string $cookie_trusted_device = 'wp_sll_trusted_device';

    private array $default_options = array(
        'max_retries'        => 5,
        'lockout_minutes'    => 15,
        'lockout_increase'   => 2,
        'whitelist_ips'      => '',
        'blacklist_ips'      => '',
        'email_enabled'      => 0,
        'email_on_lockout'   => 1,
        'notify_email'       => '',
        'disable_xmlrpc'     => 1,
        'hide_login_errors'  => 1,
        'behind_proxy'       => 0,
        'security_headers'   => 1,
        'block_user_enum'    => 1,
        'disable_file_edit'  => 1,
        'honeypot_enabled'   => 1,
        'ht_block_config'    => 0,
        'ht_disable_indexes' => 0,
        'ht_block_sensitive' => 0,
        'ht_protect_htaccess'=> 0,
        'ht_block_authorscan'=> 0,
        'hide_wp_login'      => 0,
        'custom_login_slug'  => 'panel', 
        'custom_branding'    => 0,
        'brand_logo_url'     => '', 
        'brand_bg_color'     => '#f0f2f5',
        'magic_links_active' => 0,
        'twofa_enabled'      => 1,
        'geoip_enabled'      => 0,
        'geoip_cache_hours'  => 24,
        'country_block_mode' => 'off',
        'country_allow_list' => '',
        'country_block_list' => '',
        'country_block_scope'=> 'login',
        'recaptcha_enabled'         => 0,
        'recaptcha_site_key'        => '',
        'recaptcha_secret_key'      => '',
        'recaptcha_on_login'        => 0,
        'recaptcha_on_register'     => 1,
        'recaptcha_on_lostpassword' => 1,
        'recaptcha_on_comments'     => 1,
        // --- YENİ EKLENEN AYARLAR ---
        'disable_wlwmanifest' => 1, // wlwmanifest.xml engelle
        'remove_wp_version'   => 1, // WordPress sürümünü gizle
        'remove_script_ver'   => 1, // CSS/JS versiyonlarını temizle (Divi gizleme)
        'block_double_slash'  => 1  // //site gibi taramaları engelle
    );

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'register_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'wp_dashboard_setup', array( $this, 'dashboard_widget_init' ) );

        add_filter( 'authenticate', array( $this, 'check_login_limit' ), 20, 3 );
        add_action( 'wp_login_failed', array( $this, 'record_failed_login' ) );
        add_action( 'login_form', array( $this, 'add_honeypot_field' ) );
        add_filter( 'authenticate', array( $this, 'check_honeypot' ), 10, 3 );
        
        // Hardening features (Geliştirildi)
        add_action( 'init', array( $this, 'hardening_features' ) );
        
        add_filter( 'login_errors', array( $this, 'maybe_hide_login_errors' ) );
        add_action( 'send_headers', array( $this, 'add_security_headers' ) );
        
        add_action( 'template_redirect', array( $this, 'block_user_enumeration' ) );
        add_filter( 'rest_endpoints', array( $this, 'disable_rest_user_endpoints' ) );

        add_filter( 'map_meta_cap', array( $this, 'disallow_file_edit' ), 10, 2 );
        add_action( 'update_option_' . $this->option_key, array( $this, 'update_htaccess_rules' ), 10, 2 );

        add_action( 'init', array( $this, 'handle_hide_login' ) ); // Magic link öncesi çalışması için init priority standart
        add_filter( 'site_url', array( $this, 'filter_site_url' ), 10, 4 );
        add_filter( 'wp_redirect', array( $this, 'filter_wp_redirect' ), 10, 2 );

        add_action( 'login_enqueue_scripts', array( $this, 'custom_login_design' ) );
        
        // Magic Login - Yüksek öncelik
        add_action( 'init', array( $this, 'handle_magic_login' ), 1 );

        add_action( 'admin_init', array( $this, 'handle_log_clear' ) );
        add_action( 'init', array( $this, 'apply_country_block' ), 2 );

        add_action( 'wp_login', array( $this, 'record_success_login' ), 10, 2 );
        add_action( 'login_form', array( $this, 'render_twofa_field' ) );

        // Güvenlik Akışı
        add_filter( 'wp_authenticate_user', array( $this, 'verify_recaptcha_login' ), 20, 2 );
        add_filter( 'wp_authenticate_user', array( $this, 'handle_two_factor_auth' ), 30, 2 );

        // reCAPTCHA Entegrasyonları
        add_action( 'login_enqueue_scripts', array( $this, 'enqueue_recaptcha_script' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_recaptcha_script' ) );

        add_action( 'login_form', array( $this, 'render_recaptcha_login' ) );
        add_action( 'register_form', array( $this, 'render_recaptcha_register' ) );
        add_action( 'lostpassword_form', array( $this, 'render_recaptcha_lostpassword' ) );
        
        add_filter( 'registration_errors', array( $this, 'verify_recaptcha_registration' ), 10, 3 );
        add_action( 'lostpassword_post', array( $this, 'verify_recaptcha_lostpassword' ) );

        add_action( 'comment_form_after_fields', array( $this, 'render_recaptcha_comment' ) );
        add_action( 'comment_form_logged_in_after', array( $this, 'render_recaptcha_comment' ) );
        add_filter( 'preprocess_comment', array( $this, 'verify_recaptcha_comment' ) );

        add_action( 'show_user_profile', array( $this, 'render_user_twofa_settings' ) );
        add_action( 'edit_user_profile', array( $this, 'render_user_twofa_settings' ) );
        add_action( 'personal_options_update', array( $this, 'save_user_twofa_settings' ) );
        add_action( 'edit_user_profile_update', array( $this, 'save_user_twofa_settings' ) );
    }

    // --- MAGIC LINK ---
    public function create_magic_link_url() {
        if ( ! current_user_can( 'manage_options' ) ) return false;
        $token = bin2hex( random_bytes(32) );
        $user_id = get_current_user_id();
        $hash = hash('sha256', $token);
        update_user_meta( $user_id, 'sll_magic_token', array( 'hash' => $hash, 'expires' => time() + 3600 ));
        $login_url = site_url( 'wp-login.php' );
        return add_query_arg( array('sll_magic' => $token, 'uid' => $user_id), $login_url );
    }

    public function handle_magic_login() {
        if ( isset( $_GET['sll_magic'], $_GET['uid'] ) && !is_user_logged_in() ) {
            $token = sanitize_text_field( $_GET['sll_magic'] );
            $uid   = intval( $_GET['uid'] );
            $ip    = $this->get_user_ip();
            $attempt_key = 'sll_magic_attempt_' . $ip;
            $attempts = (int) get_transient( $attempt_key );
            if ( $attempts > 5 ) { wp_die( 'Çok fazla hatalı deneme.', 'Erişim Engellendi', array('response' => 429) ); }
            
            $stored_data = get_user_meta( $uid, 'sll_magic_token', true );
            $is_valid = false;
            if ( is_array($stored_data) && !empty($stored_data['hash']) ) {
                if ( time() <= $stored_data['expires'] ) {
                    if ( hash_equals( $stored_data['hash'], hash('sha256', $token) ) ) { $is_valid = true; }
                }
            }
            if ( $is_valid ) {
                if ( isset($_POST['sll_magic_confirm']) ) {
                    delete_user_meta( $uid, 'sll_magic_token' );
                    wp_set_auth_cookie( $uid );
                    $this->log_event( $ip, 'magic_link_user', 'magic_login', 'Başarılı magic link girişi' );
                    wp_safe_redirect( admin_url() ); exit;
                } else { $this->render_magic_confirm_page(); exit; }
            } else {
                set_transient( $attempt_key, $attempts + 1, 300 );
                $this->log_event( $ip, 'unknown', 'magic_failed', 'Geçersiz magic link' );
                wp_die('<h1>Geçersiz Link</h1><p>Bu bağlantı geçersiz.</p>', 'Hata', array('response'=>403));
            }
        }
    }

    private function render_magic_confirm_page() {
        $opts = $this->get_options();
        $bg_color = !empty($opts['brand_bg_color']) ? $opts['brand_bg_color'] : '#f0f2f5';
        echo '<!DOCTYPE html><html lang="tr"><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width, initial-scale=1.0"><title>Giriş Onayı</title><style>body { font-family: sans-serif; background: ' . esc_attr($bg_color) . '; display: flex; align-items: center; justify-content: center; height: 100vh; margin: 0; } .card { background: white; padding: 40px; border-radius: 12px; box-shadow: 0 10px 25px rgba(0,0,0,0.1); text-align: center; max-width: 400px; width: 90%; } .btn { background: #2563eb; color: white; border: none; padding: 12px 30px; border-radius: 6px; cursor: pointer; width: 100%; font-size: 16px; font-weight:600; }</style></head><body><div class="card"><h1>Güvenli Giriş</h1><p>Magic Link doğrulandı.</p><form method="post"><button type="submit" name="sll_magic_confirm" value="1" class="btn">Giriş Yap</button></form></div></body></html>';
    }

    // --- reCAPTCHA Render Helper ---
    public function enqueue_recaptcha_script() {
        $opts = $this->get_options();
        if ( ! empty( $opts['recaptcha_enabled'] ) && ! empty( $opts['recaptcha_site_key'] ) ) {
            wp_enqueue_script( 'google-recaptcha', 'https://www.google.com/recaptcha/api.js', array(), null, true );
            echo '<style>.g-recaptcha { margin-bottom: 15px; }</style>';
        }
    }
    private function render_recaptcha_box_generic() {
        $opts = $this->get_options();
        if ( ! empty( $opts['recaptcha_enabled'] ) && ! empty( $opts['recaptcha_site_key'] ) ) {
            echo '<div class="g-recaptcha" data-sitekey="' . esc_attr( $opts['recaptcha_site_key'] ) . '"></div>';
        }
    }
    public function render_recaptcha_login() { $opts = $this->get_options(); if ( ! empty( $opts['recaptcha_on_login'] ) ) $this->render_recaptcha_box_generic(); }
    public function render_recaptcha_register() { $opts = $this->get_options(); if ( ! empty( $opts['recaptcha_on_register'] ) ) $this->render_recaptcha_box_generic(); }
    public function render_recaptcha_lostpassword() { $opts = $this->get_options(); if ( ! empty( $opts['recaptcha_on_lostpassword'] ) ) $this->render_recaptcha_box_generic(); }
    public function render_recaptcha_comment() { $opts = $this->get_options(); if ( ! empty( $opts['recaptcha_on_comments'] ) ) $this->render_recaptcha_box_generic(); }

    // --- reCAPTCHA Doğrulama ---
    private function verify_recaptcha_remote() {
        $opts = $this->get_options();
        if ( empty( $opts['recaptcha_enabled'] ) || empty( $opts['recaptcha_secret_key'] ) ) return true;
        $token = isset( $_POST['g-recaptcha-response'] ) ? $_POST['g-recaptcha-response'] : '';
        if ( empty( $token ) ) return new WP_Error( 'recaptcha_missing', __( 'Lütfen robot olmadığınızı doğrulayın.', 'wp-security-fortress' ) );
        $response = wp_remote_post( 'https://www.google.com/recaptcha/api/siteverify', array( 'body' => array( 'secret' => $opts['recaptcha_secret_key'], 'response' => $token, 'remoteip' => $this->get_user_ip() ) ) );
        if ( is_wp_error( $response ) ) return true; 
        $result = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( isset( $result['success'] ) && $result['success'] ) return true;
        return new WP_Error( 'recaptcha_failed', __( 'Doğrulama başarısız.', 'wp-security-fortress' ) );
    }
    public function verify_recaptcha_login( $user, $password ) {
        if ( is_wp_error( $user ) ) return $user;
        $opts = $this->get_options(); if ( empty( $opts['recaptcha_on_login'] ) ) return $user;
        $check = $this->verify_recaptcha_remote();
        if ( is_wp_error( $check ) ) { $this->log_event( $this->get_user_ip(), 'unknown', 'recaptcha_fail', 'Giriş reCAPTCHA hatası' ); return $check; }
        return $user;
    }
    public function verify_recaptcha_registration( $errors, $sanitized_user_login, $user_email ) {
        $opts = $this->get_options(); if ( empty( $opts['recaptcha_on_register'] ) ) return $errors;
        $check = $this->verify_recaptcha_remote();
        if ( is_wp_error( $check ) ) { $errors->add( 'recaptcha_error', $check->get_error_message() ); $this->log_event( $this->get_user_ip(), 'unknown', 'recaptcha_reg_fail', 'Kayıt reCAPTCHA hatası' ); }
        return $errors;
    }
    public function verify_recaptcha_lostpassword() {
        $opts = $this->get_options(); if ( empty( $opts['recaptcha_on_lostpassword'] ) ) return;
        $check = $this->verify_recaptcha_remote(); if ( is_wp_error( $check ) ) wp_die( $check->get_error_message(), 'Hata', array( 'response' => 403 ) );
    }
    public function verify_recaptcha_comment( $commentdata ) {
        $opts = $this->get_options(); if ( empty( $opts['recaptcha_on_comments'] ) ) return $commentdata;
        if ( current_user_can( 'moderate_comments' ) ) return $commentdata;
        $check = $this->verify_recaptcha_remote(); if ( is_wp_error( $check ) ) wp_die( $check->get_error_message() . ' <a href="javascript:history.back()">Geri Dön</a>', 'reCAPTCHA Hatası' );
        return $commentdata;
    }

    // --- Standart Güvenlik Fonksiyonları ---
    public function handle_log_clear() {
        if ( isset($_GET['sll_action']) && $_GET['sll_action'] === 'clear_logs' && isset($_GET['_wpnonce']) ) {
            if ( check_admin_referer('sll_clear_logs') && current_user_can('manage_options') ) {
                update_option( $this->logs_option_key, array(), 'no' );
                wp_safe_redirect( admin_url('admin.php?page=wp-security-fortress&msg=logs_cleared') );
                exit;
            }
        }
    }

    public function handle_hide_login() {
        $opts = $this->get_options();
        if ( empty( $opts['hide_wp_login'] ) || empty( $opts['custom_login_slug'] ) ) return;
        $slug = sanitize_title( $opts['custom_login_slug'] );
        $request_uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        if ( strpos( $request_uri, 'admin-ajax.php' ) !== false || (isset($_REQUEST['action']) && $_REQUEST['action'] === 'logout') ) return;
        if ( !is_user_logged_in() && strpos( $request_uri, '/wp-admin' ) !== false ) { if ( strpos( $request_uri, 'admin-ajax.php' ) === false ) { wp_safe_redirect( home_url() ); exit; } }
        if ( strpos( $request_uri, '/' . $slug ) !== false && !strpos( $request_uri, 'wp-login.php' ) ) { $_SERVER['SCRIPT_NAME'] = '/wp-login.php'; require_once ABSPATH . 'wp-login.php'; die; }
        if ( strpos( $request_uri, 'wp-login.php' ) !== false && !is_user_logged_in() && !isset( $_REQUEST['action'] ) ) { wp_safe_redirect( home_url() ); exit; }
    }

    public function filter_site_url( $url, $path, $scheme, $blog_id ) { return $this->rewrite_login_url( $url, $scheme ); }
    public function filter_wp_redirect( $location, $status ) { return $this->rewrite_login_url( $location ); }
    
    private function rewrite_login_url( $url, $scheme = null ) {
        $opts = $this->get_options();
        if ( ! empty( $opts['hide_wp_login'] ) && ! empty( $opts['custom_login_slug'] ) ) {
            $slug = sanitize_title( $opts['custom_login_slug'] );
            if ( strpos( $url, 'wp-login.php' ) !== false ) $url = str_replace( 'wp-login.php', $slug, $url );
        }
        return $url;
    }

    public function custom_login_design() {
        $opts = $this->get_options();
        if ( empty( $opts['custom_branding'] ) ) return;
        $bg_color = sanitize_hex_color( $opts['brand_bg_color'] );
        $logo_url = esc_url( $opts['brand_logo_url'] );
        echo '<style type="text/css">body.login { background-color: ' . $bg_color . ' !important; }';
        if ( ! empty( $logo_url ) ) echo '.login h1 a { background-image: url(' . $logo_url . ') !important; width: 100% !important; background-size: contain !important; height: 80px !important; }';
        echo '.login form { border-radius: 12px; box-shadow: 0 10px 25px rgba(0,0,0,0.1); border:none; padding: 40px; } .wp-core-ui .button-primary { background: #2563eb !important; border-color: #2563eb !important; border-radius: 6px; font-size:16px; padding: 5px 20px; transition:0.3s; } .wp-core-ui .button-primary:hover { background: #1e40af !important; } .login .message, .login .success { border-left-color: #10b981; border-radius: 4px; box-shadow: 0 2px 5px rgba(0,0,0,0.05); }</style>';
    }

    private function get_htaccess_path() { return ABSPATH . '.htaccess'; }
    public function sync_htaccess_state() {
        if( !is_admin() ) return;
        $path = $this->get_htaccess_path();
        if( !file_exists($path) || !is_readable($path) ) return;
        $content = file_get_contents($path);
        if( empty($content) ) return;
        $opts = $this->get_options();
        $changed = false;
        $patterns = array( 'ht_disable_indexes' => '/Options\s+.*-Indexes/i', 'ht_block_config' => '/<Files\s+["\']?wp-config\.php["\']?>.*?Deny\s+from\s+all/is', 'ht_block_sensitive' => '/<FilesMatch\s+.*(readme|license|log).*?>.*?Deny\s+from\s+all/is', 'ht_protect_htaccess' => '/<FilesMatch\s+["\']?\^.*\.ht["\']?>.*?Deny\s+from\s+all/is', 'ht_block_authorscan' => '/RewriteCond\s+%\{QUERY_STRING\}\s+\^?author=([0-9]*)/i' );
        foreach( $patterns as $key => $pattern ) { if( preg_match($pattern, $content) && empty($opts[$key]) ) { $opts[$key] = 1; $changed = true; } }
        if( $changed ) update_option( $this->option_key, $opts );
    }
    
    private function get_user_ip() {
        $opts = $this->get_options();
        if ( !empty($opts['behind_proxy']) ) {
            if ( isset($_SERVER['HTTP_CF_CONNECTING_IP']) ) { $ip = sanitize_text_field($_SERVER['HTTP_CF_CONNECTING_IP']); if ( filter_var($ip, FILTER_VALIDATE_IP) ) return $ip; }
            if ( isset($_SERVER['HTTP_X_FORWARDED_FOR']) ) { $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']); $ip = sanitize_text_field(trim($ips[0])); if ( filter_var($ip, FILTER_VALIDATE_IP) ) return $ip; }
        }
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field($_SERVER['REMOTE_ADDR']) : '0.0.0.0';
        return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : '0.0.0.0';
    }

    private function geoip_lookup( $ip ) {
        $ip = trim( (string) $ip ); if ( $ip === '' || $ip === '0.0.0.0' ) return array();
        $opts = $this->get_options(); $cache_hours = isset( $opts['geoip_cache_hours'] ) ? (int) $opts['geoip_cache_hours'] : 24;
        $cache_key = 'sll_geo_' . md5( $ip ); $cached = get_transient( $cache_key );
        if ( is_array( $cached ) && ! empty( $cached['country_code'] ) ) return $cached;
        $response = wp_remote_get( 'https://ipwho.is/' . rawurlencode( $ip ), array( 'timeout' => 3 ) );
        if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) return array();
        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! is_array( $data ) || ( isset( $data['success'] ) && ! $data['success'] ) ) return array();
        $result = array( 'ip' => $ip, 'country' => isset( $data['country'] ) ? (string) $data['country'] : '', 'country_code' => isset( $data['country_code'] ) ? strtoupper( (string) $data['country_code'] ) : '', 'city' => isset( $data['city'] ) ? (string) $data['city'] : '' );
        set_transient( $cache_key, $result, $cache_hours * HOUR_IN_SECONDS );
        return $result;
    }

    // --- 2FA Helpers ---
    private function twofa_generate_secret( $length = 16 ) {
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567'; $secret = '';
        for ( $i = 0; $i < $length; $i++ ) { $secret .= $alphabet[ random_int( 0, strlen( $alphabet ) - 1 ) ]; }
        return $secret;
    }
    private function twofa_base32_decode( $b32 ) {
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567'; $b32 = strtoupper( preg_replace( '/[^A-Z2-7]/', '', $b32 ) );
        $bits = ''; $bytes = '';
        for ( $i = 0, $len = strlen( $b32 ); $i < $len; $i++ ) { $val = strpos( $alphabet, $b32[ $i ] ); if ( $val === false ) continue; $bits .= str_pad( decbin( $val ), 5, '0', STR_PAD_LEFT ); }
        for ( $i = 0, $len = strlen( $bits ); $i + 8 <= $len; $i += 8 ) { $bytes .= chr( bindec( substr( $bits, $i, 8 ) ) ); }
        return $bytes;
    }
    private function twofa_verify_code( $secret, $code, $discrepancy = 1, $period = 30 ) {
        $code = trim( $code ); if ( ! ctype_digit( $code ) || strlen( $code ) < 6 || strlen( $code ) > 8 ) return false;
        $secret_key = $this->twofa_base32_decode( $secret ); if ( empty( $secret_key ) ) return false;
        $time_slice = floor( time() / $period ); $code = (int) $code;
        for ( $i = -$discrepancy; $i <= $discrepancy; $i++ ) {
            $slice = $time_slice + $i; $time = pack( 'N*', 0 ) . pack( 'N*', $slice );
            $hash = hash_hmac( 'sha1', $time, $secret_key, true ); $offset = ord( substr( $hash, -1 ) ) & 0x0F;
            $truncated_hash = ( ( ( ord( $hash[ $offset ] ) & 0x7F ) << 24 ) | ( ( ord( $hash[ $offset + 1 ] ) & 0xFF ) << 16 ) | ( ( ord( $hash[ $offset + 2 ] ) & 0xFF ) << 8 ) | ( ( ord( $hash[ $offset + 3 ] ) & 0xFF ) ) );
            if ( $truncated_hash % 1000000 === $code ) return true;
        }
        return false;
    }

    public function render_twofa_field() {
        $opts = $this->get_options();
        if ( empty( $opts['twofa_enabled'] ) ) return;
        echo '<p><label for="sll_2fa_code">2FA Doğrulama Kodu<br /><input type="text" name="sll_2fa_code" id="sll_2fa_code" class="input" size="20" autocomplete="one-time-code" /></label></p><p><label><input type="checkbox" name="sll_remember_2fa" value="1" /> Bu cihazı 30 gün hatırla (Güvenli)</label></p>';
    }

    public function render_user_twofa_settings( $user ) {
        $opts = $this->get_options(); if ( empty( $opts['twofa_enabled'] ) ) return;
        $secret = get_user_meta( $user->ID, 'sll_2fa_secret', true ); $enabled = get_user_meta( $user->ID, 'sll_2fa_enabled', true );
        if ( empty( $secret ) ) { $secret = $this->twofa_generate_secret(); update_user_meta( $user->ID, 'sll_2fa_secret', $secret ); }
        $site_name = get_bloginfo( 'name', 'display' );
        $otpauth = sprintf( 'otpauth://totp/%s:%s?secret=%s&issuer=%s&digits=6&period=30', rawurlencode( $site_name ), rawurlencode( $user->user_login ), $secret, rawurlencode( $site_name ) );
        $qr_url = 'https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=' . rawurlencode( $otpauth );
        echo '<h2>WP Security Fortress - 2FA</h2><table class="form-table"><tr><th scope="row">2FA Durumu</th><td><label><input type="checkbox" name="sll_2fa_enabled" value="1" ' . checked( $enabled, 1, false ) . ' /> 2FA Zorunlu</label></td></tr><tr><th scope="row">Gizli Anahtar</th><td><code>' . esc_html( $secret ) . '</code></td></tr><tr><th scope="row">QR Kod</th><td><img src="' . esc_url( $qr_url ) . '" alt="QR" /></td></tr><tr><th scope="row">Test</th><td><input type="text" name="sll_2fa_test_code" class="regular-text" /></td></tr></table>';
    }

    public function save_user_twofa_settings( $user_id ) {
        if ( ! current_user_can( 'edit_user', $user_id ) ) return;
        $opts = $this->get_options(); if ( empty( $opts['twofa_enabled'] ) ) return;
        $enabled = isset( $_POST['sll_2fa_enabled'] ) ? 1 : 0; update_user_meta( $user_id, 'sll_2fa_enabled', $enabled );
        $secret = get_user_meta( $user_id, 'sll_2fa_secret', true );
        if ( empty( $secret ) ) { $secret = $this->twofa_generate_secret(); update_user_meta( $user_id, 'sll_2fa_secret', $secret ); }
        if ( ! empty( $_POST['sll_2fa_test_code'] ) ) {
            $test = sanitize_text_field( wp_unslash( $_POST['sll_2fa_test_code'] ) );
            if ( ! $this->twofa_verify_code( $secret, $test ) ) add_action( 'user_profile_update_errors', function( $errors ) { $errors->add( 'sll_2fa_test_failed', '2FA test kodu geçersiz.' ); } );
        }
    }

    public function handle_two_factor_auth( $user, $password ) {
        $opts = $this->get_options(); if ( empty( $opts['twofa_enabled'] ) ) return $user;
        if ( is_wp_error( $user ) || ! $user instanceof WP_User ) return $user;
        $secret = get_user_meta( $user->ID, 'sll_2fa_secret', true ); $enabled = get_user_meta( $user->ID, 'sll_2fa_enabled', true );
        if ( empty( $enabled ) || empty( $secret ) ) return $user;
        if ( $this->is_trusted_device( $user->ID ) ) return $user;
        $ip = $this->get_user_ip(); $code = isset( $_POST['sll_2fa_code'] ) ? sanitize_text_field( wp_unslash( $_POST['sll_2fa_code'] ) ) : '';
        if ( $code === '' ) { $this->log_event( $ip, $user->user_login, '2fa_missing', '2FA kodu girilmedi' ); return new WP_Error( 'two_factor_required', '2FA kodu zorunlu.' ); }
        if ( ! $this->twofa_verify_code( $secret, $code ) ) { $this->log_event( $ip, $user->user_login, '2fa_failed', 'Geçersiz 2FA' ); return new WP_Error( 'two_factor_invalid', '2FA geçersiz.' ); }
        if ( isset( $_POST['sll_remember_2fa'] ) ) $this->set_trusted_device_cookie( $user->ID );
        $this->log_event( $ip, $user->user_login, '2fa_ok', '2FA doğrulandı' );
        return $user;
    }
    
    private function set_trusted_device_cookie( $user_id ) {
        $days = 30; $expire = time() + ( DAY_IN_SECONDS * $days ); $key = wp_salt('auth'); 
        $hash = hash_hmac( 'sha256', $user_id . '|' . $expire, $key );
        $cookie_val = $user_id . '|' . $expire . '|' . $hash;
        setcookie( $this->cookie_trusted_device, $cookie_val, array( 'expires' => $expire, 'path' => COOKIEPATH, 'domain' => COOKIE_DOMAIN, 'secure' => is_ssl(), 'httponly' => true, 'samesite' => 'Lax' ) );
    }
    
    private function is_trusted_device( $user_id ) {
        if ( ! isset( $_COOKIE[$this->cookie_trusted_device] ) ) return false;
        $parts = explode( '|', $_COOKIE[$this->cookie_trusted_device] ); if ( count( $parts ) !== 3 ) return false;
        list( $cookie_uid, $cookie_expire, $cookie_hash ) = $parts;
        if ( (int)$cookie_uid !== (int)$user_id ) return false; if ( time() > $cookie_expire ) return false;
        $key = wp_salt('auth'); $check_hash = hash_hmac( 'sha256', $cookie_uid . '|' . $cookie_expire, $key );
        return hash_equals( $check_hash, $cookie_hash );
    }

    private function get_options() { $opts = get_option( $this->option_key, array() ); return wp_parse_args( $opts, $this->default_options ); }
    private function parse_ip_list( $raw ) { $lines = preg_split('/[\r\n]+/', $raw); $ips = array(); if ( is_array($lines) ) { foreach ( $lines as $line ) { $parts = explode('#', $line); $ip = trim($parts[0]); if ( filter_var($ip, FILTER_VALIDATE_IP) ) $ips[] = $ip; } } return array_unique($ips); }
    private function parse_country_list( $raw ) { $codes = array(); $lines = preg_split( '/[\r\n,]+/', (string) $raw ); if ( is_array( $lines ) ) { foreach ( $lines as $line ) { $code = strtoupper( trim( $line ) ); if ( $code !== '' && preg_match( '/^[A-Z]{2}$/', $code ) ) $codes[] = $code; } } return array_unique( $codes ); }

    public function apply_country_block() {
        $opts = $this->get_options(); if ( empty( $opts['geoip_enabled'] ) ) return;
        $mode = $opts['country_block_mode'] ?? 'off'; $scope = $opts['country_block_scope'] ?? 'login'; if ( $mode === 'off' ) return;
        $ip = $this->get_user_ip(); if ( isset( $opts['whitelist_ips'] ) && in_array( $ip, $this->parse_ip_list( $opts['whitelist_ips'] ), true ) ) return;
        if ( 'login' === $scope ) { $req = $_SERVER['REQUEST_URI'] ?? ''; if ( strpos( $req, 'wp-login.php' ) === false ) return; }
        $geo = $this->geoip_lookup( $ip ); if ( empty( $geo['country_code'] ) ) return;
        $cc = strtoupper( $geo['country_code'] );
        $allowed = $this->parse_country_list( $opts['country_allow_list'] ?? '' ); $blocked = $this->parse_country_list( $opts['country_block_list'] ?? '' );
        $should_block = false;
        if ( 'whitelist' === $mode ) { if ( ! empty( $allowed ) && ! in_array( $cc, $allowed, true ) ) $should_block = true; } 
        elseif ( 'blacklist' === $mode ) { if ( ! empty( $blocked ) && in_array( $cc, $blocked, true ) ) $should_block = true; }
        if ( $should_block ) { $this->log_event( $ip, 'unknown', 'country_block', "Ülke Engeli: $cc" ); wp_die( 'Erişim Engellendi', 'Yasak', array( 'response' => 403 ) ); }
    }

    private function log_event( $ip, $username, $type, $message = '' ) {
        $logs = get_option( $this->logs_option_key, array() ); if ( !is_array($logs) ) $logs = array();
        array_unshift( $logs, array( 'time' => time(), 'ip' => substr( sanitize_text_field($ip), 0, 45 ), 'username' => substr( sanitize_user($username), 0, 50 ), 'type' => substr( sanitize_text_field($type), 0, 50 ), 'message' => substr( sanitize_text_field($message), 0, 200 ) ) );
        if ( count($logs) > $this->max_logs ) $logs = array_slice( $logs, 0, $this->max_logs );
        update_option( $this->logs_option_key, $logs, 'no' );
    }
    private function get_logs() { return get_option( $this->logs_option_key, array() ); }
    private function get_ip_transient_key( $ip ) { return 'sll_limit_' . md5($ip); }
    
    public function check_login_limit( $user, $username, $password ) {
        if ( is_wp_error($user) || empty($username) ) return $user;
        $ip = $this->get_user_ip(); $opts = $this->get_options();
        if ( in_array($ip, $this->parse_ip_list($opts['whitelist_ips']), true) ) return $user;
        if ( in_array($ip, $this->parse_ip_list($opts['blacklist_ips']), true) ) { $this->log_event( $ip, $username, 'blocked_blacklist', 'Blacklist IP' ); return new WP_Error( 'blacklisted', 'Erişim yasak.' ); }
        $transient_key = $this->get_ip_transient_key($ip); $data = get_transient($transient_key);
        if ( $data && isset($data['locked_until']) && $data['locked_until'] > time() ) {
            $rem = ceil( ($data['locked_until'] - time()) / 60 );
            $this->log_event( $ip, $username, 'blocked_lockout', "Kilitli ($rem dk)" );
            return new WP_Error( 'too_many_tries', "Çok fazla deneme. $rem dakika bekleyin." );
        }
        return $user;
    }
    
    public function record_failed_login( $username ) {
        $ip = $this->get_user_ip(); $opts = $this->get_options();
        if ( in_array($ip, $this->parse_ip_list($opts['whitelist_ips']), true) ) return;
        $transient_key = $this->get_ip_transient_key($ip); $data = get_transient($transient_key);
        if ( !$data ) $data = array( 'retries' => 0, 'lockout_count' => 0 );
        $data['retries']++; $this->log_event( $ip, $username, 'failed', 'Hatalı şifre' );
        if ( $data['retries'] >= (int) $opts['max_retries'] ) {
            $data['lockout_count']++; $data['retries'] = 0;
            $duration = (int) $opts['lockout_minutes'] * 60;
            $seconds = min( $duration * pow( max(1, (int) $opts['lockout_increase']), max(0, $data['lockout_count'] - 1) ), 86400 );
            $data['locked_until'] = time() + $seconds;
            $this->log_event( $ip, $username, 'lockout', "Kilitlendi " . ceil($seconds/60) . " dk" );
            if ( !empty($opts['email_enabled']) && !empty($opts['email_on_lockout']) ) {
                $to = !empty($opts['notify_email']) ? $opts['notify_email'] : get_option('admin_email');
                wp_mail( $to, "IP Kilitlendi: $ip", "Kullanıcı: $username\nIP: $ip" );
            }
            set_transient( $transient_key, $data, $seconds );
        } else { set_transient( $transient_key, $data, 43200 ); }
    }

    public function record_success_login( $user_login, $user ) { $this->log_event( $this->get_user_ip(), $user_login, 'login_success', 'Başarılı giriş' ); }
    public function add_honeypot_field() { if ( !empty($this->get_options()['honeypot_enabled']) ) echo '<p style="display:none;"><label>Email<input type="text" name="sll_honey_trap" value="" tabindex="-1" autocomplete="off" /></label></p>'; }
    public function check_honeypot( $user, $username, $password ) {
        if ( !empty($this->get_options()['honeypot_enabled']) && !empty($_POST['sll_honey_trap']) ) {
            $this->log_event( $this->get_user_ip(), $username, 'blocked_bot', 'Honeypot' ); sleep(2); return new WP_Error( 'spam_detected', 'Spam.' );
        } return $user;
    }
    public function add_security_headers() { if ( !empty($this->get_options()['security_headers']) && !headers_sent() ) { header('X-Content-Type-Options: nosniff'); header('X-Frame-Options: SAMEORIGIN'); header('X-XSS-Protection: 1; mode=block'); header('Referrer-Policy: strict-origin-when-cross-origin'); } }
    public function block_user_enumeration() { if ( !empty($this->get_options()['block_user_enum']) && !is_admin() && isset($_REQUEST['author']) && preg_match('/\d/', $_REQUEST['author']) ) { $this->log_event( $this->get_user_ip(), 'unknown', 'blocked_enum', 'User enum' ); wp_die( 'Forbidden', 'Forbidden', array('response' => 403) ); } }
    public function disable_rest_user_endpoints( $endpoints ) { if ( ! is_user_logged_in() ) { if ( isset( $endpoints['/wp/v2/users'] ) ) unset( $endpoints['/wp/v2/users'] ); if ( isset( $endpoints['/wp/v2/users/(?P<id>[\d]+)'] ) ) unset( $endpoints['/wp/v2/users/(?P<id>[\d]+)'] ); } return $endpoints; }
    public function disallow_file_edit( $caps, $cap ) { if ( !empty($this->get_options()['disable_file_edit']) && in_array($cap, array('edit_themes', 'edit_plugins', 'edit_files')) ) $caps[] = 'do_not_allow'; return $caps; }
    public function maybe_hide_login_errors( $error ) { return ( !empty($this->get_options()['hide_login_errors']) ) ? 'Giriş bilgileri geçersiz.' : $error; }
    public function update_htaccess_rules( $old_value, $new_value ) {
        $rules = array(); if ( !empty($new_value['ht_block_config']) ) { $rules[] = "<Files wp-config.php>"; $rules[] = "order allow,deny"; $rules[] = "deny from all"; $rules[] = "</Files>"; }
        if ( !empty($new_value['ht_disable_indexes']) ) $rules[] = "Options -Indexes"; if ( !empty($new_value['ht_block_sensitive']) ) { $rules[] = "<FilesMatch \"\.(txt|md|exe|sh|bak|inc|pot|po|mo|log|ini|sql)$\">"; $rules[] = "Order allow,deny"; $rules[] = "Deny from all"; $rules[] = "</FilesMatch>"; }
        if ( !empty($new_value['ht_protect_htaccess']) ) { $rules[] = "<FilesMatch \"^.*\.ht\">"; $rules[] = "Order allow,deny"; $rules[] = "Deny from all"; $rules[] = "</FilesMatch>"; } if ( !empty($new_value['ht_block_authorscan']) ) { $rules[] = "<IfModule mod_rewrite.c>"; $rules[] = "RewriteEngine On"; $rules[] = "RewriteCond %{QUERY_STRING} ^author=([0-9]*) [NC]"; $rules[] = "RewriteRule .* - [F]"; $rules[] = "</IfModule>"; }
        require_once( ABSPATH . 'wp-admin/includes/misc.php' ); insert_with_markers( $this->get_htaccess_path(), 'WP_Security_Rules', $rules );
    }

    // --- YENİ EKLENEN FONKSİYONLAR (BOT & VERSİYON GİZLEME) ---
    public function hardening_features() {
        $opts = $this->get_options();

        // 1. XML-RPC Koruması
        if ( !empty($opts['disable_xmlrpc']) && defined('XMLRPC_REQUEST') && XMLRPC_REQUEST ) {
            status_header(403);
            die('XML-RPC Devre Dışı');
        }

        // 2. wlwmanifest.xml Engelleme (Bot Taraması)
        if ( !empty($opts['disable_wlwmanifest']) ) {
            remove_action('wp_head', 'wlwmanifest_link'); 
            if ( isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], 'wlwmanifest.xml') !== false ) {
                $this->log_event( $this->get_user_ip(), 'bot', 'blocked_wlw', 'wlwmanifest.xml taraması' );
                status_header(403);
                die('Erişim Engellendi (WLW)');
            }
        }

        // 3. Çift Slash (//) Koruması
        if ( !empty($opts['block_double_slash']) ) {
            if ( isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '//') !== false ) {
                $this->log_event( $this->get_user_ip(), 'bot', 'blocked_scan', 'Çift slash (//) taraması' );
                status_header(403);
                die('Geçersiz İstek');
            }
        }

        // 4. WordPress Versiyon Bilgisini Gizle
        if ( !empty($opts['remove_wp_version']) ) {
            remove_action('wp_head', 'wp_generator');
        }

        // 5. CSS/JS Dosyalarından Versiyonları Kaldır (Divi Gizleme)
        if ( !empty($opts['remove_script_ver']) ) {
            add_filter( 'style_loader_src', array( $this, 'remove_ver_query_string' ), 9999 );
            add_filter( 'script_loader_src', array( $this, 'remove_ver_query_string' ), 9999 );
        }
    }

    // Yardımcı: Versiyon temizleyici
    public function remove_ver_query_string( $src ) {
        if ( strpos( $src, 'ver=' ) ) {
            $src = remove_query_arg( 'ver', $src );
        }
        return $src;
    }

    public function register_settings_page() { add_menu_page('Security Fortress','Security Fortress','manage_options','wp-security-fortress',array($this,'settings_page_html'),'dashicons-shield-alt',30); }
    public function register_settings() { register_setting('wp_sll_group',$this->option_key,array($this,'sanitize_options')); }
    public function sanitize_options( $input ) {
        $output = $this->get_options();
        foreach( array('max_retries', 'lockout_minutes', 'lockout_increase', 'geoip_cache_hours') as $field ) if ( isset($input[$field]) ) $output[$field] = absint($input[$field]);
        foreach( array('whitelist_ips', 'blacklist_ips', 'country_allow_list', 'country_block_list') as $field ) if ( isset($input[$field]) ) $output[$field] = sanitize_textarea_field($input[$field]);
        if ( isset($input['notify_email']) ) $output['notify_email'] = sanitize_email($input['notify_email']);
        if ( isset($input['custom_login_slug']) ) { $slug = sanitize_title($input['custom_login_slug']); if ( preg_match('/^[a-z0-9-]+$/', $slug) && !in_array($slug, array('wp-admin', 'admin', 'login', 'wp-login')) ) $output['custom_login_slug'] = $slug; }
        if ( isset($input['brand_logo_url']) ) $output['brand_logo_url'] = esc_url_raw($input['brand_logo_url']);
        if ( isset($input['brand_bg_color']) ) { $color = sanitize_hex_color($input['brand_bg_color']); if ( $color ) $output['brand_bg_color'] = $color; }
        if ( isset( $input['recaptcha_site_key'] ) ) $output['recaptcha_site_key'] = sanitize_text_field( $input['recaptcha_site_key'] );
        if ( isset( $input['recaptcha_secret_key'] ) ) $output['recaptcha_secret_key'] = sanitize_text_field( $input['recaptcha_secret_key'] );
        // Yeni ayarları buraya ekledik
        $checkboxes = array('email_enabled', 'email_on_lockout', 'disable_xmlrpc', 'hide_login_errors','behind_proxy', 'security_headers', 'block_user_enum', 'disable_file_edit','honeypot_enabled', 'ht_block_config', 'ht_disable_indexes', 'ht_block_sensitive','ht_protect_htaccess', 'ht_block_authorscan', 'hide_wp_login', 'custom_branding','magic_links_active', 'twofa_enabled', 'geoip_enabled', 'recaptcha_enabled', 'recaptcha_on_login', 'recaptcha_on_register', 'recaptcha_on_lostpassword', 'recaptcha_on_comments', 'disable_wlwmanifest', 'remove_wp_version', 'remove_script_ver', 'block_double_slash');
        foreach( $checkboxes as $checkbox ) $output[$checkbox] = isset($input[$checkbox]) ? 1 : 0;
        if ( isset( $input['country_block_mode'] ) ) $output['country_block_mode'] = $input['country_block_mode'];
        if ( isset( $input['country_block_scope'] ) ) $output['country_block_scope'] = $input['country_block_scope'];
        return $output;
    }

    public function settings_page_html() {
        if ( !current_user_can('manage_options') ) return;
        $this->sync_htaccess_state();
        if ( isset($_GET['msg']) && $_GET['msg'] === 'logs_cleared' ) echo '<div class="notice notice-success"><p>Loglar temizlendi.</p></div>';
        $magic_link = '';
        if ( isset($_GET['sll_action']) && $_GET['sll_action'] === 'gen_magic' && check_admin_referer('sll_magic_gen') ) $magic_link = $this->create_magic_link_url();
        $opts = $this->get_options(); 
        ?>
        <style>
            :root{--sll-primary:#2563eb;--sll-success:#10b981;--sll-text:#334155;--sll-border:#e2e8f0}
            .wp-sll-wrap{font-family:'Inter',system-ui,sans-serif;max-width:1200px;margin:20px auto;color:var(--sll-text)}
            .sll-hero{background:linear-gradient(135deg,#0f172a 0%,#334155 100%);color:#fff;padding:25px;border-radius:12px;margin-bottom:25px;display:flex;justify-content:space-between;align-items:center}
            .sll-hero h1{color:#fff;margin:0;font-size:22px}
            .nav-tab-wrapper{border-bottom:1px solid var(--sll-border)!important;margin-bottom:25px!important}
            .nav-tab{border:none!important;background:transparent!important;color:#64748b!important;font-weight:600;font-size:14px;padding:10px 20px!important}
            .nav-tab-active{color:var(--sll-primary)!important;border-bottom:2px solid var(--sll-primary)!important}
            .wp-sll-tab-content{display:none}.wp-sll-tab-content.active{display:block}
            .sll-grid{display:grid;grid-template-columns:repeat(auto-fill,minmax(320px,1fr));gap:20px}
            .sll-card{background:#fff;border:1px solid var(--sll-border);border-radius:10px;padding:20px;box-shadow:0 1px 2px rgba(0,0,0,0.05)}
            .sll-card-header{font-size:15px;font-weight:700;margin-bottom:15px;padding-bottom:10px;border-bottom:1px solid var(--sll-border);display:flex;align-items:center;gap:8px}
            .sll-item{display:flex;justify-content:space-between;align-items:center;padding:12px 0;border-bottom:1px solid #f1f5f9}
            .sll-label strong{display:block;font-size:13px;color:#1e293b}.sll-label span{font-size:11px;color:#64748b}
            .sll-switch{position:relative;display:inline-block;width:40px;height:22px}.sll-switch input{opacity:0;width:0;height:0}
            .sll-slider{position:absolute;cursor:pointer;top:0;left:0;right:0;bottom:0;background-color:#cbd5e1;transition:.3s;border-radius:22px}
            .sll-slider:before{position:absolute;content:"";height:16px;width:16px;left:3px;bottom:3px;background-color:white;transition:.3s;border-radius:50%}
            .sll-switch input:checked+.sll-slider{background-color:var(--sll-success)}.sll-switch input:checked+.sll-slider:before{transform:translateX(18px)}
            .sll-input-group label{display:block;font-weight:600;margin-bottom:6px;font-size:12px}
            .sll-input-group input[type="text"],.sll-input-group input[type="number"],.sll-input-group input[type="password"],.sll-input-group input[type="email"]{width:100%;padding:8px;border:1px solid var(--sll-border);border-radius:6px}
            .sll-btn-primary{background:var(--sll-primary);color:#fff;border:none;padding:10px 20px;border-radius:6px;cursor:pointer;font-weight:600}
            .sll-btn-outline{background:transparent;border:1px solid var(--sll-primary);color:var(--sll-primary);padding:8px 16px;border-radius:6px;text-decoration:none;font-weight:600;font-size:13px;display:inline-block}
            .sll-alert{padding:15px;border-radius:6px;margin-bottom:15px}
            .sll-alert-success{background:#dcfce7;color:#166534;border-left:3px solid #10b981}
            .sll-alert-warning{background:#fef3c7;color:#92400e;border-left:3px solid #f59e0b}
        </style>

        <div class="wrap wp-sll-wrap">
            <div class="sll-hero">
                <div><h1>WP Security Fortress <span style="font-size:12px;background:rgba(255,255,255,0.2);padding:2px 8px;border-radius:12px;">v7.5.0</span></h1></div>
                <span class="dashicons dashicons-shield-alt" style="font-size:40px;height:40px;width:40px;"></span>
            </div>

            <h2 class="nav-tab-wrapper">
                <a href="#general" class="nav-tab nav-tab-active" onclick="switchTab(event,'general')">Giriş</a>
                <a href="#design" class="nav-tab" onclick="switchTab(event,'design')">Markalama</a>
                <a href="#plesk" class="nav-tab" onclick="switchTab(event,'plesk')">Sistem</a>
                <a href="#phpsec" class="nav-tab" onclick="switchTab(event,'phpsec')">Hardening</a>
                <a href="#logs" class="nav-tab" onclick="switchTab(event,'logs')">Loglar</a>
            </h2>

            <form method="post" action="options.php">
                <?php settings_fields('wp_sll_group'); ?>
                
                <div id="general" class="wp-sll-tab-content active">
                    <div class="sll-grid">
                        <div class="sll-card">
                            <div class="sll-card-header"><span class="dashicons dashicons-admin-network"></span> Limit Ayarları</div>
                            <div class="sll-input-group" style="margin-bottom:10px;"><label>Deneme Hakkı</label><input type="number" name="<?php echo $this->option_key; ?>[max_retries]" value="<?php echo esc_attr($opts['max_retries']); ?>" min="1" max="20"></div>
                            <div class="sll-input-group" style="margin-bottom:10px;"><label>Kilit Süresi (Dk)</label><input type="number" name="<?php echo $this->option_key; ?>[lockout_minutes]" value="<?php echo esc_attr($opts['lockout_minutes']); ?>" min="5" max="1440"></div>
                            <div class="sll-input-group"><label>Kilit Artış Çarpanı</label><input type="number" name="<?php echo $this->option_key; ?>[lockout_increase]" value="<?php echo esc_attr($opts['lockout_increase']); ?>" min="1" max="5"></div>
                        </div>
                        <div class="sll-card">
                            <div class="sll-card-header"><span class="dashicons dashicons-email"></span> Bildirim</div>
                            <div class="sll-item"><div class="sll-label"><strong>Email Bildirimi</strong><span>Saldırıda haber ver.</span></div><label class="sll-switch"><input type="checkbox" name="<?php echo $this->option_key; ?>[email_enabled]" value="1" <?php checked(1, $opts['email_enabled']); ?>><span class="sll-slider"></span></label></div>
                            <div class="sll-input-group" style="margin-top:10px;"><label>Alıcı Email</label><input type="email" name="<?php echo $this->option_key; ?>[notify_email]" value="<?php echo esc_attr($opts['notify_email']); ?>" placeholder="<?php echo esc_attr(get_option('admin_email')); ?>"></div>
                        </div>
                        <div class="sll-card" style="grid-column:1/-1;">
                            <div class="sll-card-header"><span class="dashicons dashicons-admin-generic"></span> IP Listeleri</div>
                            <div class="sll-input-group" style="margin-bottom:15px;">
                                <label>Beyaz Liste</label>
                                <textarea name="<?php echo $this->option_key; ?>[whitelist_ips]" rows="3" style="width:100%;padding:8px;border:1px solid var(--sll-border);border-radius:6px;font-family:monospace;"><?php echo esc_textarea($opts['whitelist_ips']); ?></textarea>
                            </div>
                            <div class="sll-input-group">
                                <label>Kara Liste</label>
                                <textarea name="<?php echo $this->option_key; ?>[blacklist_ips]" rows="3" style="width:100%;padding:8px;border:1px solid var(--sll-border);border-radius:6px;font-family:monospace;"><?php echo esc_textarea($opts['blacklist_ips']); ?></textarea>
                            </div>
                        </div>
                        <div class="sll-card" style="grid-column:1/-1;">
                            <div class="sll-card-header"><span class="dashicons dashicons-location"></span> GeoIP &amp; Ülke Bazlı Erişim</div>
                            <div class="sll-input-group" style="margin-bottom:15px;">
                                <label><input type="checkbox" name="<?php echo $this->option_key; ?>[geoip_enabled]" value="1" <?php checked( ! empty( $opts['geoip_enabled'] ), 1 ); ?> /> GeoIP desteğini aç</label>
                            </div>
                            <div class="sll-input-group" style="display:flex;gap:10px;">
                                <div style="flex:1;"><label>Mod</label><select name="<?php echo $this->option_key; ?>[country_block_mode]" style="width:100%;"><option value="off" <?php selected( $opts['country_block_mode'], 'off' ); ?>>Kapalı</option><option value="whitelist" <?php selected( $opts['country_block_mode'], 'whitelist' ); ?>>Sadece izin verilenler</option><option value="blacklist" <?php selected( $opts['country_block_mode'], 'blacklist' ); ?>>Sadece engellenenler</option></select></div>
                                <div style="flex:1;"><label>Kapsam</label><select name="<?php echo $this->option_key; ?>[country_block_scope]" style="width:100%;"><option value="login" <?php selected( $opts['country_block_scope'], 'login' ); ?>>Sadece giriş</option><option value="site" <?php selected( $opts['country_block_scope'], 'site' ); ?>>Tüm site</option></select></div>
                                <div style="width:150px;"><label>GeoIP Cache (Saat)</label><input type="number" min="1" name="<?php echo $this->option_key; ?>[geoip_cache_hours]" value="<?php echo esc_attr( $opts['geoip_cache_hours'] ); ?>" style="width:100%;" /></div>
                            </div>
                            <div class="sll-input-group" style="margin-top:15px;"><label>İzin verilen ülkeler</label><textarea name="<?php echo $this->option_key; ?>[country_allow_list]" rows="2" style="width:100%;"><?php echo esc_textarea( $opts['country_allow_list'] ); ?></textarea></div>
                            <div class="sll-input-group"><label>Engellenen ülkeler</label><textarea name="<?php echo $this->option_key; ?>[country_block_list]" rows="2" style="width:100%;"><?php echo esc_textarea( $opts['country_block_list'] ); ?></textarea></div>
                        </div>
                    </div>
                </div>

                <div id="design" class="wp-sll-tab-content">
                    <div class="sll-grid">
                        <div class="sll-card">
                            <div class="sll-card-header"><span class="dashicons dashicons-hidden"></span> URL Gizleme</div>
                            <div class="sll-item"><div class="sll-label"><strong>Login URL Değiştir</strong></div><label class="sll-switch"><input type="checkbox" name="<?php echo $this->option_key; ?>[hide_wp_login]" value="1" <?php checked(1, $opts['hide_wp_login']); ?>><span class="sll-slider"></span></label></div>
                            <div class="sll-input-group" style="margin-top:15px;"><label>Yeni Slug</label><input type="text" name="<?php echo $this->option_key; ?>[custom_login_slug]" value="<?php echo esc_attr($opts['custom_login_slug']); ?>" placeholder="panel">
                            <?php if(!empty($opts['hide_wp_login'])): ?><div class="sll-alert sll-alert-warning" style="margin-top:10px;"><code><?php echo home_url($opts['custom_login_slug']); ?></code></div><?php endif; ?></div>
                        </div>
                        <div class="sll-card">
                            <div class="sll-card-header"><span class="dashicons dashicons-art"></span> Tasarım</div>
                            <div class="sll-item"><div class="sll-label"><strong>Özel Markalama</strong></div><label class="sll-switch"><input type="checkbox" name="<?php echo $this->option_key; ?>[custom_branding]" value="1" <?php checked(1, $opts['custom_branding']); ?>><span class="sll-slider"></span></label></div>
                            <div class="sll-input-group" style="margin-top:15px;"><label>Logo URL</label><input type="text" name="<?php echo $this->option_key; ?>[brand_logo_url]" value="<?php echo esc_attr($opts['brand_logo_url']); ?>"></div>
                            <div class="sll-input-group" style="margin-top:10px;"><label>Arka Plan Rengi</label><input type="text" name="<?php echo $this->option_key; ?>[brand_bg_color]" value="<?php echo esc_attr($opts['brand_bg_color']); ?>"></div>
                        </div>
                        <div class="sll-card" style="grid-column:1/-1;">
                            <div class="sll-card-header"><span class="dashicons dashicons-admin-links"></span> Magic Link</div>
                            <?php if($magic_link): ?><div class="sll-alert sll-alert-success"><code><?php echo esc_url($magic_link); ?></code></div><?php endif; ?>
                            <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=wp-security-fortress&sll_action=gen_magic'),'sll_magic_gen'); ?>" class="sll-btn-outline">Yeni Link Oluştur</a>
                        </div>
                    </div>
                </div>

                <div id="plesk" class="wp-sll-tab-content">
                    <div class="sll-card">
                        <div class="sll-card-header"><span class="dashicons dashicons-media-code"></span> .htaccess Kuralları</div>
                        <?php $ht=array('ht_block_config'=>'wp-config.php Koruması','ht_disable_indexes'=>'Dizin Listeleme Kapat','ht_block_sensitive'=>'Hassas Dosyaları Gizle','ht_protect_htaccess'=>'.htaccess Koruması','ht_block_authorscan'=>'Kullanıcı Tarama Engeli');
                        foreach($ht as $k=>$v): ?><div class="sll-item"><div class="sll-label"><strong><?php echo $v; ?></strong></div><label class="sll-switch"><input type="checkbox" name="<?php echo $this->option_key; ?>[<?php echo $k; ?>]" value="1" <?php checked(1,$opts[$k]); ?>><span class="sll-slider"></span></label></div><?php endforeach; ?>
                    </div>
                </div>

                <div id="phpsec" class="wp-sll-tab-content">
                    <div class="sll-grid">
                        <div class="sll-card">
                            <div class="sll-card-header"><span class="dashicons dashicons-lock"></span> Güvenlik Sıkılaştırma</div>
                            <?php 
                            // YENİ ÖZELLİKLER BURAYA EKLENDİ
                            $ph = array(
                                'twofa_enabled'       => '2FA (Google Authenticator)',
                                'disable_xmlrpc'      => 'XML-RPC Kapat',
                                'disable_wlwmanifest' => 'wlwmanifest.xml Engelle (Bot Önlemi)', // YENİ
                                'block_double_slash'  => 'Çift Slash (//) Taramasını Engelle', // YENİ
                                'remove_wp_version'   => 'WP Sürümünü Gizle', // YENİ
                                'remove_script_ver'   => 'CSS/JS Versiyonlarını Gizle (Divi Koruması)', // YENİ
                                'hide_login_errors'   => 'Login Hatalarını Gizle',
                                'disable_file_edit'   => 'Dosya Editörü Kapat',
                                'honeypot_enabled'    => 'Honeypot',
                                'security_headers'    => 'Güvenlik Başlıkları',
                                'block_user_enum'     => 'User Enum Engeli',
                                'behind_proxy'        => 'Proxy/CDN Modu'
                            );
                            foreach($ph as $k=>$v): ?><div class="sll-item"><div class="sll-label"><strong><?php echo $v; ?></strong></div><label class="sll-switch"><input type="checkbox" name="<?php echo $this->option_key; ?>[<?php echo $k; ?>]" value="1" <?php checked(1,$opts[$k]); ?>><span class="sll-slider"></span></label></div><?php endforeach; ?>
                        </div>
                        <div class="sll-card">
                            <div class="sll-card-header"><span class="dashicons dashicons-shield"></span> Google reCAPTCHA</div>
                            <div class="sll-item"><div class="sll-label"><strong>reCAPTCHA Aktif Et</strong></div><label class="sll-switch"><input type="checkbox" name="<?php echo $this->option_key; ?>[recaptcha_enabled]" value="1" <?php checked(1, $opts['recaptcha_enabled']); ?>><span class="sll-slider"></span></label></div>
                            
                            <div style="padding:10px;background:#f8fafc;border-radius:6px;margin:10px 0;">
                                <div class="sll-item" style="border:none;padding:5px 0;"><div class="sll-label" style="font-weight:normal;">Giriş Formunda Göster (Login)</div><label class="sll-switch" style="transform:scale(0.8);"><input type="checkbox" name="<?php echo $this->option_key; ?>[recaptcha_on_login]" value="1" <?php checked(1, $opts['recaptcha_on_login']); ?>><span class="sll-slider"></span></label></div>
                                <div class="sll-item" style="border:none;padding:5px 0;"><div class="sll-label" style="font-weight:normal;">Kayıt Formunda Göster (Register)</div><label class="sll-switch" style="transform:scale(0.8);"><input type="checkbox" name="<?php echo $this->option_key; ?>[recaptcha_on_register]" value="1" <?php checked(1, $opts['recaptcha_on_register']); ?>><span class="sll-slider"></span></label></div>
                                <div class="sll-item" style="border:none;padding:5px 0;"><div class="sll-label" style="font-weight:normal;">Şifre Sıfırlamada Göster</div><label class="sll-switch" style="transform:scale(0.8);"><input type="checkbox" name="<?php echo $this->option_key; ?>[recaptcha_on_lostpassword]" value="1" <?php checked(1, $opts['recaptcha_on_lostpassword']); ?>><span class="sll-slider"></span></label></div>
                                <div class="sll-item" style="border:none;padding:5px 0;"><div class="sll-label" style="font-weight:normal;">Yorum Formunda Göster</div><label class="sll-switch" style="transform:scale(0.8);"><input type="checkbox" name="<?php echo $this->option_key; ?>[recaptcha_on_comments]" value="1" <?php checked(1, $opts['recaptcha_on_comments']); ?>><span class="sll-slider"></span></label></div>
                            </div>

                            <div class="sll-input-group" style="margin-top:15px;"><label>Site Key</label><input type="text" name="<?php echo $this->option_key; ?>[recaptcha_site_key]" value="<?php echo esc_attr($opts['recaptcha_site_key']); ?>"></div>
                            <div class="sll-input-group" style="margin-top:10px;"><label>Secret Key</label><input type="password" name="<?php echo $this->option_key; ?>[recaptcha_secret_key]" value="<?php echo esc_attr($opts['recaptcha_secret_key']); ?>"></div>
                        </div>
                    </div>
                </div>
                
                <div id="logs" class="wp-sll-tab-content">
                    <div class="sll-card">
                        <div class="sll-card-header"><span><span class="dashicons dashicons-list-view"></span> Olaylar</span><a href="<?php echo wp_nonce_url(admin_url('admin.php?page=wp-security-fortress&sll_action=clear_logs'), 'sll_clear_logs'); ?>" class="sll-btn-outline" onclick="return confirm('Silinsin mi?');">Temizle</a></div>
                        <div style="overflow-x:auto;"><table style="width:100%;font-size:13px;border-collapse:collapse;"><thead><tr style="background:#f8fafc;text-align:left;"><th style="padding:10px;">Tarih</th><th>IP</th><th>Kullanıcı</th><th>Olay</th><th>Detay</th></tr></thead><tbody><?php $logs=$this->get_logs(); if(empty($logs)): ?><tr><td colspan="5" style="padding:20px;text-align:center;color:#999;">Kayıt yok.</td></tr><?php else: foreach(array_slice($logs,0,50) as $l): ?><tr><td style="padding:10px;border-bottom:1px solid #eee;"><?php echo date_i18n('d.m.Y H:i',$l['time']); ?></td><td style="padding:10px;border-bottom:1px solid #eee;"><?php echo esc_html($l['ip']); ?></td><td style="padding:10px;border-bottom:1px solid #eee;"><?php echo esc_html($l['username']); ?></td><td style="padding:10px;border-bottom:1px solid #eee;"><?php echo esc_html($l['type']); ?></td><td style="padding:10px;border-bottom:1px solid #eee;color:#666;"><?php echo esc_html($l['message']); ?></td></tr><?php endforeach; endif; ?></tbody></table></div>
                    </div>
                </div>

                <div style="margin-top:30px;"><button type="submit" class="sll-btn-primary">Değişiklikleri Kaydet</button></div>
            </form>
            <script>function switchTab(e,id){e.preventDefault();document.querySelectorAll('.nav-tab').forEach(t=>t.classList.remove('nav-tab-active'));document.querySelectorAll('.wp-sll-tab-content').forEach(c=>c.classList.remove('active'));e.target.classList.add('nav-tab-active');document.getElementById(id).classList.add('active');}</script>
        </div>
        <?php
    }
    
    public function dashboard_widget_init() {
        wp_add_dashboard_widget('wp_sll_stats', '🛡️ Security Fortress Durumu', function() {
            $logs = $this->get_logs();
            $week = time() - (7 * 24 * 60 * 60);
            
            $stats = array('honeypot' => 0, 'recaptcha' => 0, 'brute' => 0, 'geoip' => 0, 'magic' => 0);

            foreach ($logs as $l) {
                if ($l['time'] > $week) {
                    if (strpos($l['type'], 'blocked_bot') !== false) $stats['honeypot']++;
                    elseif (strpos($l['type'], 'recaptcha') !== false) $stats['recaptcha']++;
                    elseif (strpos($l['type'], 'country_block') !== false) $stats['geoip']++;
                    elseif (in_array($l['type'], array('failed', 'lockout', 'blocked_lockout'))) $stats['brute']++;
                    elseif ($l['type'] === 'magic_login') $stats['magic']++;
                }
            }

            echo '<div style="display:grid;grid-template-columns:1fr 1fr;gap:15px;margin-bottom:15px;">';
            echo '<div style="background:#f0fdf4;padding:15px;border-radius:8px;text-align:center;border:1px solid #bbf7d0;"><span class="dashicons dashicons-buddicons-replies" style="font-size:24px;color:#16a34a;width:auto;height:auto;"></span><div style="font-size:20px;font-weight:bold;color:#166534;margin-top:5px;">' . $stats['honeypot'] . '</div><div style="font-size:11px;color:#15803d;">Honeypot Avı</div></div>';
            echo '<div style="background:#fef2f2;padding:15px;border-radius:8px;text-align:center;border:1px solid #fecaca;"><span class="dashicons dashicons-warning" style="font-size:24px;color:#dc2626;width:auto;height:auto;"></span><div style="font-size:20px;font-weight:bold;color:#991b1b;margin-top:5px;">' . $stats['brute'] . '</div><div style="font-size:11px;color:#7f1d1d;">Hatalı Giriş</div></div>';
            echo '<div style="background:#eff6ff;padding:15px;border-radius:8px;text-align:center;border:1px solid #bfdbfe;"><span class="dashicons dashicons-shield" style="font-size:24px;color:#2563eb;width:auto;height:auto;"></span><div style="font-size:20px;font-weight:bold;color:#1e40af;margin-top:5px;">' . $stats['recaptcha'] . '</div><div style="font-size:11px;color:#1e3a8a;">reCAPTCHA Engel</div></div>';
            echo '<div style="background:#fffbeb;padding:15px;border-radius:8px;text-align:center;border:1px solid #fde68a;"><span class="dashicons dashicons-globe" style="font-size:24px;color:#d97706;width:auto;height:auto;"></span><div style="font-size:20px;font-weight:bold;color:#92400e;margin-top:5px;">' . $stats['geoip'] . '</div><div style="font-size:11px;color:#78350f;">Ülke Engeli</div></div>';
            echo '</div>';
            echo '<div style="border-top:1px solid #eee;padding-top:10px;display:flex;justify-content:space-between;align-items:center;"><span style="color:#64748b;font-size:12px;">Son 7 Günlük Rapor</span><a href="' . admin_url('admin.php?page=wp-security-fortress-live-traffic') . '" class="button button-small">Canlı Trafik</a></div>';
        });
    }
}

new WP_Security_Fortress();